% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_manual_bounded_constraints.R
\name{add_manual_bounded_constraints}
\alias{add_manual_bounded_constraints}
\alias{add_manual_bounded_constraints,ConservationProblem,data.frame-method}
\alias{add_manual_bounded_constraints,ConservationProblem,tbl_df-method}
\title{Add manually specified bound constraints}
\usage{
add_manual_bounded_constraints(x, data)

\S4method{add_manual_bounded_constraints}{ConservationProblem,data.frame}(x, data)

\S4method{add_manual_bounded_constraints}{ConservationProblem,tbl_df}(x, data)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} (i.e. \code{\linkS4class{ConservationProblem}}) object.}

\item{data}{\code{data.frame} or \code{\link[tibble:tibble]{tibble::tibble()}} object.
See the Data format section for more information.}
}
\value{
Object (i.e. \code{\linkS4class{ConservationProblem}}) with the constraints
added to it.
}
\description{
Add constraints to a conservation planning \code{\link[=problem]{problem()}} to ensure
that the planning unit values (e.g. proportion, binary) in a solution
range between specific lower and upper bounds. This function offers more
fine-grained control than the \code{\link[=add_manual_locked_constraints]{add_manual_locked_constraints()}}
function and is is most useful for problems involving proportion-type
or semi-continuous decisions.
}
\section{Data format}{

The argument to \code{data} should be a \code{data.frame} with the following fields
(columns):

\describe{

\item{pu}{\code{integer} planning unit identifier.}

\item{zone}{\code{character} names of zones. Note that this
argument is optional for arguments to \code{x} that contain a single
zone.}

\item{lower}{\code{numeric} values indicating the minimum
value that each planning unit can be allocated to in each zone
in the solution.}

\item{upper}{\code{numeric} values indicating the maximum
value that each planning unit can be allocated to in each zone
in the solution.}

}
}

\examples{
# set seed for reproducibility
set.seed(500)

# load data
data(sim_pu_polygons, sim_features, sim_pu_zones_polygons,
     sim_features_zones)

# create minimal problem
p1 <- problem(sim_pu_polygons, sim_features, "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)

# create problem with locked in constraints using add_locked_constraints
p2 <- p1 \%>\% add_locked_in_constraints("locked_in")

# create identical problem using add_manual_bounded_constraints
bounds_data <- data.frame(pu = which(sim_pu_polygons$locked_in),
                          lower = 1, upper = 1)

p3 <- p1 \%>\% add_manual_bounded_constraints(bounds_data)
\dontrun{
# solve problems
s1 <- solve(p1)
s2 <- solve(p2)
s3 <- solve(p3)

# plot solutions
par(mfrow = c(1,3), mar = c(0, 0, 4.1, 0))
plot(s1, main = "none locked in")
plot(s1[s1$solution_1 == 1, ], col = "darkgreen", add = TRUE)

plot(s2, main = "add_locked_in_constraints")
plot(s2[s2$solution_1 == 1, ], col = "darkgreen", add = TRUE)

plot(s3, main = "add_bounds_constraints")
plot(s3[s3$solution_1 == 1, ], col = "darkgreen", add = TRUE)
}
# create minimal problem with multiple zones
p4 <- problem(sim_pu_zones_polygons, sim_features_zones,
              c("cost_1", "cost_2", "cost_3")) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5,
                                  ncol = 3)) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)

# create data.frame with the following constraints:
# planning units 1, 2, and 3 must be allocated to zone 1 in the solution
# planning units 4, and 5 must be allocated to zone 2 in the solution
# planning units 8 and 9 must not be allocated to zone 3 in the solution
bounds_data2 <- data.frame(pu = c(1, 2, 3, 4, 5, 8, 9),
                           zone = c(rep("zone_1", 3), rep("zone_2", 2),
                                    rep("zone_3", 2)),
                           lower = c(rep(1, 5), rep(0, 2)),
                           upper = c(rep(1, 5), rep(0, 2)))

# print bounds data
print(bounds_data2)

# create problem with added constraints
p5 <- p4 \%>\% add_manual_bounded_constraints(bounds_data2)
\dontrun{
# solve problem
s4 <- solve(p4)
s5 <- solve(p5)

# create two new columns representing the zone id that each planning unit
# was allocated to in the two solutions
s4$solution <- category_vector(s4@data[, c("solution_1_zone_1",
                                           "solution_1_zone_2",
                                           "solution_1_zone_3")])
s4$solution <- factor(s4$solution)

s4$solution_bounded <- category_vector(s5@data[, c("solution_1_zone_1",
                                                   "solution_1_zone_2",
                                                   "solution_1_zone_3")])
s4$solution_bounded <- factor(s4$solution_bounded)

# plot solutions
spplot(s4, zcol = c("solution", "solution_bounded"), axes = FALSE,
       box = FALSE)
}

}
\seealso{
See \link{constraints} for an overview of all functions for adding constraints.

Other constraints: 
\code{\link{add_contiguity_constraints}()},
\code{\link{add_feature_contiguity_constraints}()},
\code{\link{add_linear_constraints}()},
\code{\link{add_locked_in_constraints}()},
\code{\link{add_locked_out_constraints}()},
\code{\link{add_mandatory_allocation_constraints,ConservationProblem-method}},
\code{\link{add_manual_locked_constraints}()}
}
\concept{constraints}
