# nolint start
# Currently, only the GET verb is used.

#' Submit a GET request to the PTV API
#'
#' @section Obtaining API authentication details:
#'
#' You will need to obtain a user ID (also called a devid) and an API key from
#' Public Transport Victoria. These are obtained by email. Instructions are
#' available at \url{https://www.ptv.vic.gov.au/footer/data-and-reporting/datasets/ptv-timetable-api/}.
#' You may pass these two pieces of information directly to the function, or you
#' can set the PTV_USER_ID and PTV_API_KEY environment variables.
#'
#' @param request A request or path for the API, eg. "routes".
#' @param user_id Integer or character. A user ID or devid provided by Public
#'   Transport Victoria. Refer to `?ptvapi` for more details.
#' @param api_key Character. An API key, with dashes, provided by Public
#'   Transport Victoria. Refer to `?ptvapi` for more details.
#' @param ... Additional arguments passed to `httr::GET`.
#'
#' @return A HTTP response. Content can be accessed with `httr::content`.
#'
#' @keywords internal
#'
PTVGET <- function(request,
                   user_id = determine_user_id(),
                   api_key = determine_api_key(),
                   ...) {
  request_url <- generate_request_url(
    request = request,
    user_id = user_id,
    api_key = api_key
  )
  request_url_without_auth <- prefix_base_url_and_version(request)
  response <- httr::GET(url = request_url, ...)
  process_response(response, request_url_without_auth)
}
# nolint end

#' Process a raw httr response and return an object of class ptv_api
#'
#' This S3 object returned by this function contains three elements:
#' \itemize{
#'   \item status_code, as an integer
#'   \item request, which is the request URL without authentication details
#'   \item content, the unparsed body of the response
#' }
#'
#' @param response A raw response generated by the `httr` package
#' @param request_url_without_auth Character. The request `url`, without `devid`
#'   and signature
#'
#' @return An S3 object of class ptv_api
#'
#' @keywords internal
#'
process_response <- function(response, request_url_without_auth) {

  status_code <- httr::status_code(response)

  if (status_code == 404) {
    stop("URL not found: ", request_url_without_auth)
  }

  content <- jsonlite::fromJSON(
    httr::content(response, "text", encoding = "UTF-8"),
    simplifyVector = FALSE
  )

  if (status_code == 400) {
    stop("Invalid request: ", request_url_without_auth, " - ", content$message)
  }
  if (status_code == 403) {
    stop("Access denied.")
  }
  if (status_code != 200) {
    stop("Status code ", status_code)
  }

  list(
    request = request_url_without_auth,
    retrieved = format(
      Sys.time(),
      format = "%Y-%m-%d %H:%M:%OS %Z",
      tz = "Australia/Melbourne"
    ),
    status_code = status_code,
    content = content
  )
}
