#' Transform Codes to Start-End Times
#' 
#' Transforms the range coding structure(s) from \code{\link[qdap]{cm_time.temp}} 
#' (in list format) into a data frame of start and end times in long format.
#' 
#' @param \ldots List object(s) in the form generated by 
#' \code{\link[qdap]{cm_time.temp}}.
#' @param v.name An optional name for the column created for the list.var 
#' argument
#' @param list.var logical.  If TRUE creates a column for the data frame created 
#' by each time.list passed to \code{cm_t2l}.
#' @param start.end logical. If TRUE outputs stop and end times for each 
#' \code{\link[qdap]{cm_time.temp}} list object.
#' @param debug logical. If TRUE debugging mode is on.  
#' \code{\link[qdap]{cm_time2long}} will return possible errors in time span 
#' inputs.
#' @return Generates a dataframe of start and end times for each code.
#' @seealso 
#' \code{\link{cm_df2long}},
#' \code{\link{cm_time.temp}}
#' @references Miles, M. B. & Huberman, A. M. (1994). An expanded sourcebook: 
#' Qualitative   data analysis. 2nd ed. Thousand Oaks, CA: SAGE Publications.
#' @keywords coding, time-span
#' @export
#' @examples
#' \dontrun{
#' x <- list(
#'     transcript_time_span = qcv(00:00 - 1:12:00),
#'     A = qcv(terms = "2.40:3.00, 5.01, 6.02:7.00, 9.00"),
#'     B = qcv(terms = "2.40, 3.01:3.02, 5.01, 6.02:7.00, 
#'         9.00, 1.12.00:1.19.01"),
#'     C = qcv(terms = "2.40:3.00, 5.01, 6.02:7.00, 9.00, 17.01")
#' )
#' (dat <- cm_time2long(x))
#' gantt_wrap(dat, "code", border.color = "black", border.size = 5)
#' }
cm_time2long <-
function(..., v.name = "variable", list.var = TRUE, 
    start.end = FALSE, debug = TRUE){
    mf <- match.call(expand.dots = FALSE)
    objs <- as.character(mf[[2]])
    L1 <- lapply(objs, get)
    names(L1) <- objs
    if(debug){
        x <- suppressMessages(lapply(L1, function(x) {
            cm_debug(x)
        }))
        m <- x[!sapply(x, is.null)]
        if (!identical(as.character(m), character(0))) {
            message("Warning: possible errors found:\n")
            print(m); stop("Check warnings")
        }  
    }  
    L2 <- lapply(L1, cm_t2l, list.var = FALSE, start.end = start.end)
    if (list.var) {
        L2 <- lapply(seq_along(L2), function(i) data.frame(L2[[i]], 
            VAR = objs[i]))
    }
    DF <- data.frame(do.call(rbind, L2), row.names = NULL)
    if (list.var) {
        colnames(DF)[ncol(DF)] <- v.name
    }
    comment(DF) <- "cmtime"
    DF
}