% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/textmodel_nb.R
\name{textmodel_nb}
\alias{textmodel_nb}
\title{Naive Bayes classifier for texts}
\usage{
textmodel_nb(
  x,
  y,
  smooth = 1,
  prior = c("uniform", "docfreq", "termfreq"),
  distribution = c("multinomial", "Bernoulli")
)
}
\arguments{
\item{x}{the \link[quanteda:dfm]{dfm} on which the model will be fit.  Does not
need to contain only the training documents.}

\item{y}{vector of training labels associated with each document identified
in \code{train}.  (These will be converted to factors if not already
factors.)}

\item{smooth}{smoothing parameter for feature counts, added to the
feature frequency totals by training class}

\item{prior}{prior distribution on texts; one of \code{"uniform"},
\code{"docfreq"}, or \code{"termfreq"}.  See Prior Distributions below.}

\item{distribution}{count model for text features, can be \code{multinomial} or
\code{Bernoulli}.  To fit a "binary multinomial" model, first convert the dfm to
a binary matrix using \verb{[quanteda::dfm_weight](x, scheme = "boolean")}.}
}
\value{
\code{textmodel_nb()} returns a list consisting of the following (where
\eqn{I} is the total number of documents, \eqn{J} is the total number of
features, and \eqn{k} is the total number of training classes):

\item{call}{original function call}

\item{param}{\eqn{k \times V}; class conditional posterior estimates}

\item{x}{the \eqn{N \times V} training dfm \code{x}}

\item{y}{the \eqn{N}-length \code{y} training class vector, where NAs will
not be used will be retained in the saved \code{x} matrix}

\item{distribution}{character; the distribution of \code{x} for the NB
model}

\item{priors}{numeric; the class prior probabilities}

\item{smooth}{numeric; the value of the smoothing parameter}
}
\description{
Fit a multinomial or Bernoulli Naive Bayes model, given a dfm and some
training labels.
}
\section{Prior distributions}{


Prior distributions refer to the prior probabilities assigned to the
training classes, and the choice of prior distribution affects the
calculation of the fitted probabilities.  The default is uniform priors,
which sets the unconditional probability of observing the one class to be
the same as observing any other class.

"Document frequency" means that the class priors will be taken from the
relative proportions of the class documents used in the training set.  This
approach is so common that it is assumed in many examples, such as the
worked example from Manning, Raghavan, and Schütze (2008) below.  It is not
the default in \pkg{quanteda}, however, since there may be nothing
informative in the relative numbers of documents used to train a classifier
other than the relative availability of the documents.  When training
classes are balanced in their number of documents (usually advisable),
however, then the empirically computed "docfreq" would be equivalent to
"uniform" priors.

Setting \code{prior} to "termfreq" makes the priors equal to the proportions of
total feature counts found in the grouped documents in each training class,
so that the classes with the largest number of features are assigned the
largest priors. If the total count of features in each training class was
the same, then "uniform" and "termfreq" would be the same.
}

\section{Smoothing parameter}{


The \code{smooth} value is added to the feature frequencies, aggregated by
training class, to avoid zero frequencies in any class.  This has the
effect of giving more weight to infrequent term occurrences.
}

\examples{
## Example from 13.1 of _An Introduction to Information Retrieval_
library("quanteda")
txt <- c(d1 = "Chinese Beijing Chinese",
         d2 = "Chinese Chinese Shanghai",
         d3 = "Chinese Macao",
         d4 = "Tokyo Japan Chinese",
         d5 = "Chinese Chinese Chinese Tokyo Japan")
x <- dfm(tokens(txt), tolower = FALSE)
y <- factor(c("Y", "Y", "Y", "N", NA), ordered = TRUE)

## replicate IIR p261 prediction for test set (document 5)
(tmod1 <- textmodel_nb(x, y, prior = "docfreq"))
summary(tmod1)
coef(tmod1)
predict(tmod1, type = "prob")
predict(tmod1)

# contrast with other priors
predict(textmodel_nb(x, y, prior = "uniform"))
predict(textmodel_nb(x, y, prior = "termfreq"))

## replicate IIR p264 Bernoulli Naive Bayes
tmod2 <- textmodel_nb(x, y, distribution = "Bernoulli", prior = "docfreq")
predict(tmod2, newdata = x[5, ], type = "prob")
predict(tmod2, newdata = x[5, ])
}
\references{
Manning, C.D., Raghavan, P., & Schütze, H. (2008). \emph{An
Introduction to Information Retrieval}. Cambridge: Cambridge University
Press (Chapter 13). Available at
\url{https://nlp.stanford.edu/IR-book/pdf/irbookonlinereading.pdf}.

Jurafsky, D. & Martin, J.H. (2018). From \emph{Speech and Language Processing:
An Introduction to Natural Language Processing, Computational Linguistics,
and Speech Recognition}. Draft of September 23, 2018 (Chapter 6, Naive
Bayes). Available at \url{https://web.stanford.edu/~jurafsky/slp3/}.
}
\seealso{
\code{\link[=predict.textmodel_nb]{predict.textmodel_nb()}}
}
\author{
Kenneth Benoit
}
