% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tokens_compound.R
\name{tokens_compound}
\alias{tokens_compound}
\title{Convert token sequences into compound tokens}
\usage{
tokens_compound(
  x,
  pattern,
  valuetype = c("glob", "regex", "fixed"),
  concatenator = concat(x),
  window = 0L,
  case_insensitive = TRUE,
  join = TRUE,
  apply_if = NULL
)
}
\arguments{
\item{x}{an input \link{tokens} object}

\item{pattern}{a character vector, list of character vectors, \link{dictionary},
or collocations object.  See \link{pattern} for details.}

\item{valuetype}{the type of pattern matching: \code{"glob"} for "glob"-style
wildcard expressions; \code{"regex"} for regular expressions; or \code{"fixed"} for
exact matching. See \link{valuetype} for details.}

\item{concatenator}{character; the concatenation character that will connect
the tokens making up a multi-token sequence.}

\item{window}{integer; a vector of length 1 or 2 that specifies size of the
window of tokens adjacent to \code{pattern} that will be compounded with matches
to \code{pattern}.  The window can be asymmetric if two elements are specified,
with the first giving the window size before \code{pattern} and the second the
window size after.  If paddings (empty \code{""} tokens) are found, window will
be shrunk to exclude them.}

\item{case_insensitive}{logical; if \code{TRUE}, ignore case when matching a
\code{pattern} or \link{dictionary} values}

\item{join}{logical; if \code{TRUE}, join overlapping compounds into a single
compound; otherwise, form these separately.  See examples.}

\item{apply_if}{logical vector of length \code{ndoc(x)}; documents are modified
only when corresponding values are \code{TRUE}, others are left unchanged.}
}
\value{
A \link{tokens} object in which the token sequences matching \code{pattern}
have been replaced by new compounded "tokens" joined by the concatenator.
}
\description{
Replace multi-token sequences with a multi-word, or "compound" token.  The
resulting compound tokens will represent a phrase or multi-word expression,
concatenated with \code{concatenator} (by default, the "\verb{_}" character) to form a
single "token".  This ensures that the sequences will be processed
subsequently as single tokens, for instance in constructing a \link{dfm}.
}
\note{
Patterns to be compounded (naturally) consist of multi-word sequences,
and how these are expected in \code{pattern} is very specific.  If the elements
to be compounded are supplied as space-delimited elements of a character
vector, wrap the vector in \code{\link[=phrase]{phrase()}}.  If the elements to be compounded
are separate elements of a character vector, supply it as a list where each
list element is the sequence of character elements.

See the examples below.
}
\examples{
txt <- "The United Kingdom is leaving the European Union."
toks <- tokens(txt, remove_punct = TRUE)

# character vector - not compounded
tokens_compound(toks, c("United", "Kingdom", "European", "Union"))

# elements separated by spaces - not compounded
tokens_compound(toks, c("United Kingdom", "European Union"))

# list of characters - is compounded
tokens_compound(toks, list(c("United", "Kingdom"), c("European", "Union")))

# elements separated by spaces, wrapped in phrase() - is compounded
tokens_compound(toks, phrase(c("United Kingdom", "European Union")))

# supplied as values in a dictionary (same as list) - is compounded
# (keys do not matter)
tokens_compound(toks, dictionary(list(key1 = "United Kingdom",
                                      key2 = "European Union")))
# pattern as dictionaries with glob matches
tokens_compound(toks, dictionary(list(key1 = c("U* K*"))), valuetype = "glob")

# note the differences caused by join = FALSE
compounds <- list(c("the", "European"), c("European", "Union"))
tokens_compound(toks, pattern = compounds, join = TRUE)
tokens_compound(toks, pattern = compounds, join = FALSE)

# use window to form ngrams
tokens_remove(toks, pattern = stopwords("en")) |>
    tokens_compound(pattern = "leav*", join = FALSE, window = c(0, 3))

}
