% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rollcast.R
\name{rollcast}
\alias{rollcast}
\title{Rolling one-step ahead forecasts of Value at Risk and Expected Shortfall}
\usage{
rollcast(
  x,
  p = 0.975,
  model = c("EWMA", "GARCH"),
  method = c("plain", "age", "vwhs", "fhs"),
  lambda = c(0.94, 0.98),
  nout = NULL,
  nwin = NULL,
  nboot = NULL,
  smoothscale = c("none", "lpr", "auto"),
  smoothopts = list(),
  ...
)
}
\arguments{
\item{x}{a numeric vector of asset returns}

\item{p}{confidence level for VaR calculation; default is \code{0.975}}

\item{model}{model for estimating conditional volatility; options are \code{'EWMA'}
and \code{'GARCH'}; if \code{model = 'GARCH'}, additional arguments can be adjusted
via \code{...}; default is \code{'EWMA'}}

\item{method}{method to be used for calculation; default is \code{'plain'}}

\item{lambda}{decay factor for the calculation of weights; default is \code{0.98}
for \code{method = 'age'} and \code{0.94} for \code{method = 'vwhs'} or
\code{method = 'fhs'}}

\item{nout}{number of out-of-sample observations; most recent observations are used;
default is \code{NULL}}

\item{nwin}{window size for rolling one-step forecasting; most recent observations
before out-of-sample are used; default is \code{NULL}}

\item{nboot}{size of bootstrap sample; must be a single non-NA integer value
with \code{nboot > 0}; default is \code{NULL}}

\item{smoothscale}{a character object; defines the smoothing approach
for the unconditional variance from the logarithm of the squared centralized
returns; for \code{smoothscale = 'lpr'}, the unconditional
variance is smoothed via the \code{smoots::gsmooth()} function from the
\code{smoots} package; the bandwidth has to be chosen manually; otherwise the
default is used; if \code{smoothscale = 'auto'}, the function \code{smoots::msmooth()}
is employed and the bandwidth is chosen automatically (data-driven); see the
documentation of the \code{smoots} package for more information; is set to
\code{smoothscale = 'none'} by default}

\item{smoothopts}{additional arguments of \code{smoots::gsmooth()} and
\code{smoots::msmooth()}; see the documentation of the \code{smoots}
package for more information; is set to customized default settings}

\item{...}{additional arguments of the \code{ugarchspec} function from the
\code{rugarch}-package; only applied if \code{model = 'GARCH'}; default
settings for the arguments \code{variance.model} and \code{mean.model} are:
\describe{
\item{\code{variance.model} = \code{list(model = 'sGARCH', garchOrder = c(1, 1))}}{}
\item{\code{mean.model} = \code{list(armaOrder = c(0, 0))}}{}
}}
}
\value{
Returns a list with the following elements:
\describe{
\item{VaR}{Numerical vector containing out-of-sample forecasts of Value at
Risk}
\item{ES}{Numerical vector containing out-of-sample forecasts of Expected
Shortfall (Conditional Value at Risk)}
\item{xout}{Numerical vector containing out-of-sample returns}
\item{p}{Confidence level for VaR calculation}
\item{model}{Model for estimating conditional volatility}
\item{method}{Method to be used for calculation}
\item{nout}{Number of out-of-sample observations}
\item{nwin}{Window size for rolling one-step forecasting}
\item{nboot}{Size of bootstrap sample}
}
}
\description{
Computes rolling one-step ahead forecasts of Value at Risk and Expected
Shortfall (Conditional Value at Risk) by means of plain historical
simulation age- and volatility-weighted historical simulation as well as
filtered historical simulation.
}
\examples{

prices <- DAX$price.close
returns <- diff(log(prices))
n <- length(returns)
nout <- 250 # number of obs. for out-of-sample forecasting
nwin <- 500 # window size for rolling forecasts


### Example 1 - plain historical simulation
results1 <- rollcast(x = returns, p = 0.975, method = 'plain', nout = nout,
                     nwin = nwin)
matplot(1:nout, cbind(-results1$xout, results1$VaR, results1$ES),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = 'Plain HS - 97.5\% VaR and ES for the DAX30 return series')

### Example 2 - age weighted historical simulation
results2 <- rollcast(x = returns, p = 0.975, method = 'age', nout = nout,
                     nwin = nwin)
matplot(1:nout, cbind(-results2$xout, results2$VaR, results2$ES),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = 'Age weighted HS - 97.5\% VaR and ES for the DAX30 return series')

### Example 3 - volatility weighted historical simulation - EWMA
results3 <- rollcast(x = returns, p = 0.975, model = 'EWMA',
                     method = 'vwhs', nout = nout, nwin = nwin)
matplot(1:nout, cbind(-results3$xout, results3$VaR, results3$ES),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = 'Vol. weighted HS (EWMA) - 97.5\% VaR and ES for the DAX30 return
  series')
\donttest{
### Example 4 - volatility weighted historical simulation - GARCH
results4 <- rollcast(x = returns, p = 0.975, model = 'GARCH',
                     method = 'vwhs', nout = nout, nwin = nwin)
matplot(1:nout, cbind(-results4$xout, results4$VaR, results4$ES),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = 'Vol. weighted HS (GARCH) - 97.5\% VaR and ES for the DAX30 return
  series')
}
### Example 5 - filtered historical simulation - EWMA
results5 <- rollcast(x = returns, p = 0.975, model = 'EWMA',
                     method = 'fhs', nout = nout, nwin = nwin, nboot = 10000)
matplot(1:nout, cbind(-results5$xout, results5$VaR, results5$ES),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = 'Filtered HS (EWMA) - 97.5\% VaR and ES for the DAX30 return
  series')
\donttest{
### Example 6 - filtered historical simulation - GARCH
results6 <- rollcast(x = returns, p = 0.975, model = 'GARCH',
                     method = 'fhs', nout = nout, nwin = nwin, nboot = 10000)
matplot(1:nout, cbind(-results6$xout, results6$VaR, results6$ES),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = 'Filtered HS (GARCH) - 97.5\% VaR and ES for the DAX30 return
  series')
}
}
