% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/describes_functions.R
\name{covs_test}
\alias{covs_test}
\title{Covariances Test of Significance}
\usage{
covs_test(data, vrb.nm, use = "pairwise", ci.level = 0.95, rtn.dfm = FALSE)
}
\arguments{
\item{data}{data.frame of data.}

\item{vrb.nm}{character vector of colnames specifying the variables in
\code{data} to conduct the significant test of the covariances.}

\item{use}{character vector of length 1 specifying how missing values are
handled. Currently, there are only two options: 1) "pairwise" for pairwise
deletion (i.e., \code{cov(use = "pairwise.complete.obs")}), or 2)
"complete" for listwise deletion (i.e., \code{cov(use = "complete.obs")}).}

\item{ci.level}{numeric vector of length 1 specifying the confidence level.
It must be between 0 and 1 - or it can be NULL in which case confidence
intervals are not computed and the return object does not have "lwr" or
"upr" columns.}

\item{rtn.dfm}{logical vector of length 1 specifying whether the return
object should be an array (FALSE) or data.frame (TRUE). If an array, then
the first two dimensions are the matrix dimensions from the covariance
matrix and the 3rd dimension (aka layers) contains the statistical
information (e.g., est, se, t). If  data.frame, then the first two columns
are the matrix dimensions from the covariance matrix expanded and the rest
of the columns contain the statistical information (e.g., est, se, t).}
}
\value{
If \code{rtn.dfm = FALSE}, an array where its first two dimensions
are the matrix dimensions from the covariance matrix and the 3rd dimension
(aka layers) contains the statistical information detailed below. If
\code{rtn.dfm = TRUE}, a data.frame where its first two columns are the
expanded matrix dimensions from the covariance matrix and the rest of the
columns contain the statistical information detailed below:

\describe{
   \item{cov}{sample covariances}
   \item{se}{standard errors of the covariances}
   \item{t}{t-values}
   \item{df}{degrees of freedom (n - 2)}
   \item{p}{two-sided p-values}
   \item{lwr}{lower bound of the confidence intervals (excluded if \code{ci.level = NULL})}
   \item{upr}{upper bound of the confidence intervals (excluded if \code{ci.level = NULL})}
}
}
\description{
\code{covs_test} computes sample covariances and tests for their significance
with the Pearson method assuming multivariate normality of the data. Note,
the normal-theory significance test for the covariance is much more sensitive
to departures from normality than the significant test for the mean. This
function is the covariance analogue to the \code{psych::corr.test()} function
for correlations.
}
\examples{

# traditional use
covs_test(data = attitude, vrb.nm = names(attitude))
covs_test(data = attitude, vrb.nm = names(attitude),
   ci.level = NULL) # no confidence intervals
covs_test(data = attitude, vrb.nm = names(attitude),
   rtn.dfm = TRUE) # return object as data.frame

# NOT same as simple linear regression slope
covTest <- covs_test(data = attitude, vrb.nm = names(attitude),
   ci.level = NULL, rtn.dfm = TRUE)
x <- covTest[with(covTest, rownames == "rating" & colnames == "complaints"), ]
lm_obj <- lm(rating ~ complaints, data = attitude)
y <- coef(summary(lm_obj))["complaints", , drop = FALSE]
print(x); print(y)
z <- x[, "cov"] / var(attitude$"complaints")
print(z) # dividing by variance of the predictor gives you the regression slope
# but the t-values and p-values are still different

# NOT same as correlation coefficient
covTest <- covs_test(data = attitude, vrb.nm = names(attitude),
   ci.level = NULL, rtn.dfm = TRUE)
x <- covTest[with(covTest, rownames == "rating" & colnames == "complaints"), ]
cor_test <- cor.test(x = attitude[[1]], y = attitude[[2]])
print(x); print(cor_test)
z <- x[, "cov"] / sqrt(var(attitude$"rating") * var(attitude$"complaints"))
print(z) # dividing by sqrt of the variances gives you the correlation
# but the t-values and p-values are still different

}
\seealso{
\code{\link{cov}} for covariance matrix estimates,
   \code{\link[psych]{corr.test}} for correlation matrix significant testing,
}
