% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inla_rspde.R
\name{rspde.result}
\alias{rspde.result}
\title{rSPDE result extraction from INLA estimation results}
\usage{
rspde.result(
  inla,
  name,
  rspde,
  compute.summary = TRUE,
  parameterization = "detect",
  n_samples = 5000,
  n_density = 1024
)
}
\arguments{
\item{inla}{An \code{inla} object obtained from a call to
\code{inla()}.}

\item{name}{A character string with the name of the rSPDE effect
in the inla formula.}

\item{rspde}{The \code{inla_rspde} object used for the effect in
the inla formula.}

\item{compute.summary}{Should the summary be computed?}

\item{parameterization}{If 'detect', the parameterization from the model will be used. Otherwise, the options are 'spde', 'matern' and 'matern2'.}

\item{n_samples}{The number of samples to be used if parameterization is different from the one used to fit the model.}

\item{n_density}{The number of equally spaced points to estimate the density.}
}
\value{
If the model was fitted with \code{matern} parameterization (the default), it returns a list containing:
\item{marginals.range}{Marginal densities for the range parameter}
\item{marginals.log.range}{Marginal densities for log(range)}
\item{marginals.std.dev}{Marginal densities for std. deviation}
\item{marginals.log.std.dev}{Marginal densities for log(std. deviation)}
\item{marginals.values}{Marginal densities for the field values}
\item{summary.log.range}{Summary statistics for log(range)}
\item{summary.log.std.dev}{Summary statistics for log(std. deviation)}
\item{summary.values}{Summary statistics for the field values}
If \code{compute.summary} is \code{TRUE}, then the list will also contain
\item{summary.kappa}{Summary statistics for kappa}
\item{summary.tau}{Summary statistics for tau}
If the model was fitted with the \code{spde} parameterization, it returns a list containing:
\item{marginals.kappa}{Marginal densities for kappa}
\item{marginals.log.kappa}{Marginal densities for log(kappa)}
\item{marginals.log.tau}{Marginal densities for log(tau)}
\item{marginals.tau}{Marginal densities for tau}
\item{marginals.values}{Marginal densities for the field values}
\item{summary.log.kappa}{Summary statistics for log(kappa)}
\item{summary.log.tau}{Summary statistics for log(tau)}
\item{summary.values}{Summary statistics for the field values}
If \code{compute.summary} is \code{TRUE}, then the list will also contain
\item{summary.kappa}{Summary statistics for kappa}
\item{summary.tau}{Summary statistics for tau}

For both cases, if nu was estimated, then the list will also contain
\item{marginals.nu}{Marginal densities for nu}
If nu was estimated and a beta prior was used, then the list will
also contain
\item{marginals.logit.nu}{Marginal densities for logit(nu)}
\item{summary.logit.nu}{Marginal densities for logit(nu)}
If nu was estimated and a truncated lognormal prior was used,
then the list will also contain
\item{marginals.log.nu}{Marginal densities for log(nu)}
\item{summary.log.nu}{Marginal densities for log(nu)}
If nu was estimated and \code{compute.summary} is \code{TRUE},
then the list will also contain
\item{summary.nu}{Summary statistics for nu}
}
\description{
Extract field and parameter values and distributions
for an rspde effect from an inla result object.
}
\examples{
\donttest{ #tryCatch version
tryCatch({
if (requireNamespace("INLA", quietly = TRUE)) {
  library(INLA)

  set.seed(123)

  m <- 100
  loc_2d_mesh <- matrix(runif(m * 2), m, 2)
  mesh_2d <- inla.mesh.2d(
    loc = loc_2d_mesh,
    cutoff = 0.05,
    max.edge = c(0.1, 0.5)
  )
  sigma <- 1
  range <- 0.2
  nu <- 0.8
  kappa <- sqrt(8 * nu) / range
  op <- matern.operators(
    mesh = mesh_2d, nu = nu,
    range = range, sigma = sigma, m = 2,
    parameterization = "matern"
  )
  u <- simulate(op)
  A <- inla.spde.make.A(
    mesh = mesh_2d,
    loc = loc_2d_mesh
  )
  sigma.e <- 0.1
  y <- A \%*\% u + rnorm(m) * sigma.e
  Abar <- rspde.make.A(mesh = mesh_2d, loc = loc_2d_mesh)
  mesh.index <- rspde.make.index(name = "field", mesh = mesh_2d)
  st.dat <- inla.stack(
    data = list(y = as.vector(y)),
    A = Abar,
    effects = mesh.index
  )
  rspde_model <- rspde.matern(
    mesh = mesh_2d,
    nu.upper.bound = 2
  )
  f <- y ~ -1 + f(field, model = rspde_model)
  rspde_fit <- inla(f,
    data = inla.stack.data(st.dat),
    family = "gaussian",
    control.predictor =
      list(A = inla.stack.A(st.dat))
  )
  result <- rspde.result(rspde_fit, "field", rspde_model)
  summary(result)
}
#stable.tryCatch
}, error = function(e){print("Could not run the example")})
}
}
