#' Matrix representation of the order generated by a set of intervals
#'
#' Represent the order generated by a set of intervals as a boolean matrix.
#' This is a common input format for programs that operate on partial orders.
#'
#' @param intervals data frame (see [rankUncertainty::generateIntervals] for
#' the required format)
#' @param strict is this <= or <?
#' @param binary output is coded as 0/1 if TRUE and FALSE/TRUE otherwise
#'
#' @return A boolean matrix.  If strict is set to TRUE, the (i, j)th entry is
#' intervals\[i, 'right'\] < intervals\[j, 'left'\].  If strict is set to false, <=
#' is used in place of <.
#' @export
#'
#' @examples
#' intervals <- generateIntervals(10)
#' toMatrix(intervals)
#'
#' @importFrom magrittr %>%
toMatrix <- function(intervals, strict = FALSE, binary = FALSE)
{
  errorCheck(intervals, FALSE)

  f <- function(x)
  {
    i <- x[1]
    j <- x[2]
    intervals[i, 'right'] < intervals[j, 'left']
  }

  n <- nrow(intervals)
  M <- expand.grid(1:n, 1:n) %>% apply(1, f)
  M <- matrix(M, nrow = n, ncol = n)
  if (strict)
  {
    M <- M | (rep(TRUE, n) %>% diag())
  }
  if (binary)
  {
    M <- 1 * M
  }
  M
}
