% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rasterpic_img.R
\name{rasterpic_img}
\alias{rasterpic_img}
\title{Convert an image to a geo-tagged raster}
\usage{
rasterpic_img(
  x,
  img,
  halign = 0.5,
  valign = 0.5,
  expand = 0,
  crop = FALSE,
  mask = FALSE,
  inverse = FALSE,
  crs
)
}
\arguments{
\item{x}{It could be
\itemize{
\item A \code{sf}, \code{sfc}, \code{sfg} or bounding box (see \code{\link[sf:st_bbox]{sf::st_bbox()}}) object
(from the \CRANpkg{sf} package).
\item A \code{SpatRaster}, \code{SpatVector} or \code{SpatExtent} object (from the
\CRANpkg{terra} package).
\item A numeric vector of length 4 with the extent to be used for geotagging (
i.e. \code{c(xmin, ymin, xmax, ymax)}).
}}

\item{img}{An image to be geotagged. It can be a local file or an online
file (e.g. \code{"https://i.imgur.com/6yHmlwT.jpeg"}). The following image
extensions are accepted:
\itemize{
\item \code{png}
\item \code{jpeg/jpg}
\item \code{tiff/tif}
}}

\item{halign}{Horizontal alignment of \code{img} with respect to the \code{x} object.
It should be a value between \code{0} (\code{x} is aligned on the left edge of the
raster) and \code{1} (\code{x} is on the right edge of the raster).}

\item{valign}{Vertical alignment of \code{img} with respect to the \code{x} object.
It should be a value between \code{0} (\code{x} is aligned on the bottom edge of the
raster) and \code{1} (\code{x} is on the top edge of the raster).}

\item{expand}{An expansion factor of the bounding box of \code{x}. \code{0} means that
no expansion is added, \code{1} means that the bounding box is expanded to double
the original size.}

\item{crop}{Logical. Should the raster be cropped to the (expanded) bounding
box of \code{x}?}

\item{mask}{Logical. Should the raster be masked to \code{x}? See \code{\link[terra:mask]{terra::mask()}}
for details. This option is only valid if \code{x} is a \code{sf/sfc} object.}

\item{inverse}{Logical. It affects only if \code{mask = TRUE}. If \code{TRUE}, areas on
the raster that do not overlap with \code{x} are masked.}

\item{crs}{Character string describing a coordinate reference system.
This parameter would only affect if \code{x} does not present a Coordinate
Reference System (e.g. when \code{x} is a \code{SpatExtent}, \code{sfg} \code{bbox} or a
vector of coordinates). See \strong{Details}}
}
\value{
A \code{SpatRaster} object (see \code{\link[terra:rast]{terra::rast()}}) where each layer
corresponds to a color channel of the \code{img} file.
\itemize{
\item If the \code{img} has at least 3 channels (e.g. layers), the result would have
an additional property setting the layers 1 to 3 as the Red, Green and Blue
channels. See \code{\link[terra:RGB]{terra::RGB()}}.
\item If the \code{img} already has a definition or RGB values (this may be the case
for \code{tiff/tif} files) the result would keep that channel definition.
}
}
\description{
Geotags an image based on the coordinates of a given spatial object.
}
\details{
The function preserves the Coordinate Reference System of the \code{x} object. For
optimal results do not use geographic coordinates (longitude/latitude).

\code{crs} can be in a WKT format, as a "authority:number" code such as
\code{"EPSG:4326"}, or a PROJ-string format such as \code{"+proj=utm +zone=12"}. It can
be also retrieved as \code{sf::st_crs(25830)$wkt} or using
\code{\link[tidyterra:pull_crs]{tidyterra::pull_crs()}}. See \strong{Value} and \strong{Notes} on \code{\link[terra:crs]{terra::crs()}}.
}
\examples{
\donttest{
library(sf)
library(terra)

x_path <- system.file("gpkg/UK.gpkg", package = "rasterpic")
x <- st_read(x_path, quiet = TRUE)
img <- system.file("img/vertical.png", package = "rasterpic")

# Default config
ex1 <- rasterpic_img(x, img)

ex1

plot(ex1)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# Expand
ex2 <- rasterpic_img(x, img, expand = 0.5)

plot(ex2)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# Align
ex3 <- rasterpic_img(x, img, halign = 0)

plot(ex3)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# Crop
ex4 <- rasterpic_img(x, img, crop = TRUE)

plot(ex4)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# Mask
ex5 <- rasterpic_img(x, img, mask = TRUE)

plot(ex5)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# Mask inverse
ex6 <- rasterpic_img(x, img, mask = TRUE, inverse = TRUE)

plot(ex6)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# Combine Mask inverse and crop
ex7 <- rasterpic_img(x, img, crop = TRUE, mask = TRUE, inverse = TRUE)

plot(ex7)
plot(x$geom, add = TRUE, col = NA, border = "white", lwd = 2)

# RGB channels ------
ex_rgb <- ex1
has.RGB(ex_rgb)
RGB(ex_rgb)

# Modify RGB channels
RGB(ex_rgb) <- c(2, 3, 1)
RGB(ex_rgb)

plot(ex_rgb)

# Remove RGB channels
RGB(ex_rgb) <- NULL
has.RGB(ex_rgb)
RGB(ex_rgb)

# Note the difference with terra::plot
plot(ex_rgb)
}
}
\seealso{
\code{\link[sf:st_crs]{sf::st_crs()}}, \code{\link[sf:st_bbox]{sf::st_bbox()}}, \code{\link[terra:crs]{terra::crs()}},
\code{\link[terra:ext]{terra::ext()}}, \code{\link[terra:RGB]{terra::RGB()}}.
}
