/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.io;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import org.openscience.cdk.config.Elements;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemObject;
import org.openscience.cdk.interfaces.IPseudoAtom;
import org.openscience.cdk.interfaces.IStereoElement;
import org.openscience.cdk.interfaces.ITetrahedralChirality;
import org.openscience.cdk.io.DefaultChemObjectWriter;
import org.openscience.cdk.io.MDLV2000Writer;
import org.openscience.cdk.io.MDLValence;
import org.openscience.cdk.io.formats.IResourceFormat;
import org.openscience.cdk.io.formats.MDLV3000Format;
import org.openscience.cdk.io.setting.IOSetting;
import org.openscience.cdk.io.setting.StringIOSetting;
import org.openscience.cdk.sgroup.Sgroup;
import org.openscience.cdk.sgroup.SgroupBracket;
import org.openscience.cdk.sgroup.SgroupKey;
import org.openscience.cdk.sgroup.SgroupType;

public final class MDLV3000Writer
extends DefaultChemObjectWriter {
    private static final Pattern R_GRP_NUM = Pattern.compile("R(\\d+)");
    private V30LineWriter writer;
    private StringIOSetting programNameOpt;

    public MDLV3000Writer(Writer writer) {
        this();
        this.writer = new V30LineWriter(writer);
    }

    public MDLV3000Writer(OutputStream out) throws CDKException {
        this();
        this.setWriter(out);
    }

    public MDLV3000Writer() {
        this.initIOSettings();
    }

    private static int nullAsZero(Integer x) {
        return x == null ? 0 : x;
    }

    private static <T> Integer findIdx(Map<T, Integer> idxs, T obj) {
        Integer idx = idxs.get(obj);
        if (idx == null) {
            return -1;
        }
        return idx;
    }

    private String getProgName() {
        String progname = this.programNameOpt.getSetting();
        if (progname == null) {
            return "        ";
        }
        if (progname.length() > 8) {
            return progname.substring(0, 8);
        }
        if (progname.length() < 8) {
            return String.format("%-8s", progname);
        }
        return progname;
    }

    private void writeHeader(IAtomContainer mol) throws IOException {
        String title = mol.getTitle();
        if (title != null) {
            this.writer.writeDirect(title.substring(0, Math.min(80, title.length())));
        }
        this.writer.writeDirect('\n');
        this.writer.writeDirect("  ");
        this.writer.writeDirect(this.getProgName());
        this.writer.writeDirect(new SimpleDateFormat("MMddyyHHmm").format(System.currentTimeMillis()));
        int dim = this.getNumberOfDimensions(mol);
        if (dim != 0) {
            this.writer.writeDirect(Integer.toString(dim));
            this.writer.writeDirect('D');
        }
        this.writer.writeDirect('\n');
        String comment = (String)mol.getProperty("cdk:Remark");
        if (comment != null) {
            this.writer.writeDirect(comment.substring(0, Math.min(80, comment.length())));
        }
        this.writer.writeDirect('\n');
        this.writer.writeDirect("  0  0  0     0  0            999 V3000\n");
    }

    private static ITetrahedralChirality.Stereo getLocalParity(Map<IChemObject, Integer> idxs, ITetrahedralChirality stereo) {
        IAtom[] neighbours = stereo.getLigands();
        int[] neighbourIdx = new int[neighbours.length];
        assert (neighbours.length == 4);
        for (int i = 0; i < 4; ++i) {
            neighbourIdx[i] = stereo.getChiralAtom().equals(neighbours[i]) ? Integer.MAX_VALUE : idxs.get(neighbours[i]);
        }
        boolean inverted = false;
        for (int i = 0; i < 4; ++i) {
            for (int j = i + 1; j < 4; ++j) {
                if (neighbourIdx[i] <= neighbourIdx[j]) continue;
                inverted = !inverted;
            }
        }
        return inverted ? stereo.getStereo() : stereo.getStereo().invert();
    }

    private void writeAtomBlock(IAtomContainer mol, IAtom[] atoms, Map<IChemObject, Integer> idxs, Map<IAtom, ITetrahedralChirality> atomToStereo) throws IOException, CDKException {
        if (mol.getAtomCount() == 0) {
            return;
        }
        int dim = this.getNumberOfDimensions(mol);
        this.writer.write("BEGIN ATOM\n");
        int atomIdx = 0;
        for (IAtom atom : atoms) {
            ITetrahedralChirality stereo;
            Matcher matcher;
            int elem = MDLV3000Writer.nullAsZero(atom.getAtomicNumber());
            int chg = MDLV3000Writer.nullAsZero(atom.getFormalCharge());
            int mass = MDLV3000Writer.nullAsZero(atom.getMassNumber());
            int hcnt = MDLV3000Writer.nullAsZero(atom.getImplicitHydrogenCount());
            int elec = mol.getConnectedSingleElectronsCount(atom);
            int rad = 0;
            switch (elec) {
                case 1: {
                    rad = MDLV2000Writer.SPIN_MULTIPLICITY.Monovalent.getValue();
                    break;
                }
                case 2: {
                    rad = MDLV2000Writer.SPIN_MULTIPLICITY.DivalentSinglet.getValue();
                }
            }
            int expVal = 0;
            for (IBond bond : mol.getConnectedBondsList(atom)) {
                if (bond.getOrder() == null) {
                    throw new CDKException("Unsupported bond order: " + (Object)((Object)bond.getOrder()));
                }
                expVal += bond.getOrder().numeric().intValue();
            }
            String symbol = this.getSymbol(atom, elem);
            int rnum = -1;
            if (symbol.charAt(0) == 'R' && (matcher = R_GRP_NUM.matcher(symbol)).matches()) {
                symbol = "R#";
                rnum = Integer.parseInt(matcher.group(1));
            }
            this.writer.write(++atomIdx).write(' ').write(symbol).write(' ');
            Point2d p2d = atom.getPoint2d();
            Point3d p3d = atom.getPoint3d();
            switch (dim) {
                case 0: {
                    this.writer.write("0 0 0 ");
                    break;
                }
                case 2: {
                    if (p2d != null) {
                        this.writer.write(p2d.x).writeDirect(' ').write(p2d.y).writeDirect(' ').write("0 ");
                        break;
                    }
                    this.writer.write("0 0 0 ");
                    break;
                }
                case 3: {
                    if (p3d != null) {
                        this.writer.write(p3d.x).writeDirect(' ').write(p3d.y).writeDirect(' ').write(p3d.z).writeDirect(' ');
                        break;
                    }
                    this.writer.write("0 0 0 ");
                }
            }
            this.writer.write(MDLV3000Writer.nullAsZero(atom.getProperty("cdk:AtomAtomMapping", Integer.class)));
            if (chg != 0 && chg >= -15 && chg <= 15) {
                this.writer.write(" CHG=").write(chg);
            }
            if (mass != 0) {
                this.writer.write(" MASS=").write(mass);
            }
            if (rad > 0 && rad < 4) {
                this.writer.write(" RAD=").write(rad);
            }
            if (rnum >= 0) {
                this.writer.write(" RGROUPS=(1 ").write(rnum).write(")");
            }
            if (MDLValence.implicitValence(elem, chg, expVal) - expVal != hcnt) {
                int val = expVal + hcnt;
                if (val <= 0 || val > 14) {
                    val = -1;
                }
                this.writer.write(" VAL=").write(val);
            }
            if ((stereo = atomToStereo.get(atom)) != null) {
                switch (MDLV3000Writer.getLocalParity(idxs, stereo)) {
                    case CLOCKWISE: {
                        this.writer.write(" CFG=1");
                        break;
                    }
                    case ANTI_CLOCKWISE: {
                        this.writer.write(" CFG=2");
                        break;
                    }
                }
            }
            this.writer.write('\n');
        }
        this.writer.write("END ATOM\n");
    }

    private String getSymbol(IAtom atom, int elem) {
        if (atom instanceof IPseudoAtom) {
            return ((IPseudoAtom)atom).getLabel();
        }
        String symbol = Elements.ofNumber(elem).symbol();
        if (symbol.isEmpty()) {
            symbol = atom.getSymbol();
        }
        if (symbol == null) {
            symbol = "*";
        }
        if (symbol.length() > 3) {
            symbol = symbol.substring(0, 3);
        }
        return symbol;
    }

    private void writeBondBlock(IAtomContainer mol, Map<IChemObject, Integer> idxs) throws IOException, CDKException {
        if (mol.getBondCount() == 0) {
            return;
        }
        List sgroups = (List)mol.getProperty("cdk:CtabSgroups");
        HashMap<IBond, Sgroup> multicenterSgroups = new HashMap<IBond, Sgroup>();
        if (sgroups != null) {
            for (Sgroup sgroup : sgroups) {
                if (sgroup.getType() != SgroupType.ExtMulticenter) continue;
                for (IBond bond : sgroup.getBonds()) {
                    multicenterSgroups.put(bond, sgroup);
                }
            }
        }
        this.writer.write("BEGIN BOND\n");
        int bondIdx = 0;
        for (IBond bond : mol.bonds()) {
            int order;
            IAtom beg = bond.getBegin();
            IAtom end = bond.getEnd();
            if (beg == null || end == null) {
                throw new IllegalStateException("Bond " + bondIdx + " had one or more atoms.");
            }
            int begIdx = MDLV3000Writer.findIdx(idxs, beg);
            int endIdx = MDLV3000Writer.findIdx(idxs, end);
            if (begIdx < 0 || endIdx < 0) {
                throw new IllegalStateException("Bond " + bondIdx + " had atoms not present in the molecule.");
            }
            IBond.Stereo stereo = bond.getStereo();
            if (stereo == IBond.Stereo.UP_INVERTED || stereo == IBond.Stereo.DOWN_INVERTED || stereo == IBond.Stereo.UP_OR_DOWN_INVERTED) {
                int tmp = begIdx;
                begIdx = endIdx;
                endIdx = tmp;
            }
            int n = order = bond.getOrder() == null ? 0 : bond.getOrder().numeric();
            if (order < 1 || order > 3) {
                throw new CDKException("Bond order " + (Object)((Object)bond.getOrder()) + " cannot be written to V3000");
            }
            this.writer.write(++bondIdx).write(' ').write(order).write(' ').write(begIdx).write(' ').write(endIdx);
            switch (stereo) {
                case UP: 
                case UP_INVERTED: {
                    this.writer.write(" CFG=1");
                    break;
                }
                case UP_OR_DOWN: 
                case UP_OR_DOWN_INVERTED: {
                    this.writer.write(" CFG=2");
                    break;
                }
                case DOWN: 
                case DOWN_INVERTED: {
                    this.writer.write(" CFG=3");
                    break;
                }
                case NONE: {
                    break;
                }
            }
            Sgroup sgroup = (Sgroup)multicenterSgroups.get(bond);
            if (sgroup != null) {
                ArrayList<IAtom> atoms = new ArrayList<IAtom>(sgroup.getAtoms());
                atoms.remove(bond.getBegin());
                atoms.remove(bond.getEnd());
                this.writer.write(" ATTACH=ANY ENDPTS=(").write(atoms, idxs).write(')');
            }
            this.writer.write('\n');
        }
        this.writer.write("END BOND\n");
    }

    private IAtom[] pushHydrogensToBack(IAtomContainer mol, Map<IChemObject, Integer> atomToIdx) {
        assert (atomToIdx.isEmpty());
        IAtom[] atoms = new IAtom[mol.getAtomCount()];
        for (IAtom atom : mol.atoms()) {
            if (atom.getAtomicNumber() == 1) continue;
            atoms[atomToIdx.size()] = atom;
            atomToIdx.put(atom, atomToIdx.size() + 1);
        }
        for (IAtom atom : mol.atoms()) {
            if (atom.getAtomicNumber() != 1) continue;
            atoms[atomToIdx.size()] = atom;
            atomToIdx.put(atom, atomToIdx.size() + 1);
        }
        return atoms;
    }

    private List<Sgroup> getSgroups(IAtomContainer mol) {
        ArrayList sgroups = (ArrayList)mol.getProperty("cdk:CtabSgroups");
        if (sgroups == null) {
            sgroups = new ArrayList(0);
        }
        return sgroups;
    }

    private int getNumberOfDimensions(IAtomContainer mol) {
        for (IAtom atom : mol.atoms()) {
            if (atom.getPoint3d() != null) {
                return 3;
            }
            if (atom.getPoint2d() == null) continue;
            return 2;
        }
        return 0;
    }

    private void writeSgroupBlock(List<Sgroup> sgroups, Map<IChemObject, Integer> idxs) throws IOException, CDKException {
        sgroups = new ArrayList<Sgroup>(sgroups);
        Iterator<Sgroup> iter = sgroups.iterator();
        while (iter.hasNext()) {
            if (iter.next().getType() != SgroupType.ExtMulticenter) continue;
            iter.remove();
        }
        if (sgroups.isEmpty()) {
            return;
        }
        this.writer.write("BEGIN SGROUP\n");
        Collections.sort(sgroups, new Comparator<Sgroup>(){

            @Override
            public int compare(Sgroup o1, Sgroup o2) {
                int cmp = -Boolean.compare(o1.getParents().isEmpty(), o2.getParents().isEmpty());
                if (cmp != 0 || o1.getParents().isEmpty()) {
                    return cmp;
                }
                if (o1.getParents().contains(o2)) {
                    return 1;
                }
                if (o2.getParents().contains(o1)) {
                    return -1;
                }
                return 0;
            }
        });
        int sgroupIdx = 0;
        for (Sgroup sgroup : sgroups) {
            SgroupType type = sgroup.getType();
            this.writer.write(++sgroupIdx).write(' ').write(type.getKey()).write(" 0");
            if (!sgroup.getAtoms().isEmpty()) {
                this.writer.write(" ATOMS=(").write(sgroup.getAtoms(), idxs).write(")");
            }
            if (!sgroup.getBonds().isEmpty()) {
                if (type == SgroupType.CtabData) {
                    this.writer.write(" CBONDS=(");
                } else {
                    this.writer.write(" XBONDS=(");
                }
                this.writer.write(sgroup.getBonds(), idxs);
                this.writer.write(")");
            }
            if (!sgroup.getParents().isEmpty()) {
                Set<Sgroup> parents = sgroup.getParents();
                if (parents.size() > 1) {
                    throw new CDKException("Cannot write Sgroup with multiple parents");
                }
                this.writer.write(" PARENT=").write(1 + sgroups.indexOf(parents.iterator().next()));
            }
            for (SgroupKey key : sgroup.getAttributeKeys()) {
                switch (key) {
                    case CtabSubType: {
                        this.writer.write(" SUBTYPE=").write(sgroup.getValue(key).toString());
                        break;
                    }
                    case CtabConnectivity: {
                        this.writer.write(" CONNECT=").write(sgroup.getValue(key).toString().toUpperCase(Locale.ROOT));
                        break;
                    }
                    case CtabSubScript: {
                        if (type == SgroupType.CtabMultipleGroup) {
                            this.writer.write(" MULT=").write(sgroup.getValue(key).toString());
                            break;
                        }
                        this.writer.write(" LABEL=").write(sgroup.getValue(key).toString());
                        break;
                    }
                    case CtabBracketStyle: {
                        Integer btype = (Integer)sgroup.getValue(key);
                        if (!btype.equals(1)) break;
                        this.writer.write(" BRKTYP=PAREN");
                        break;
                    }
                    case CtabParentAtomList: {
                        Collection parentAtoms = (Collection)sgroup.getValue(key);
                        this.writer.write(" PATOMS=(").write(parentAtoms, idxs).write(')');
                        break;
                    }
                    case CtabComponentNumber: {
                        Integer number = (Integer)sgroup.getValue(key);
                        if (number <= 0) break;
                        this.writer.write(" COMPNO=").write(number);
                        break;
                    }
                    case CtabExpansion: {
                        boolean expanded = (Boolean)sgroup.getValue(key);
                        if (!expanded) break;
                        this.writer.write(" ESTATE=E");
                        break;
                    }
                    case CtabBracket: {
                        Collection brackets = (Collection)sgroup.getValue(key);
                        for (SgroupBracket bracket : brackets) {
                            this.writer.write(" BRKXYZ=(");
                            Point2d p1 = bracket.getFirstPoint();
                            Point2d p2 = bracket.getSecondPoint();
                            this.writer.write("9");
                            this.writer.write(' ').write(p1.x).write(' ').write(p1.y).write(" 0");
                            this.writer.write(' ').write(p2.x).write(' ').write(p2.y).write(" 0");
                            this.writer.write(" 0 0 0");
                            this.writer.write(")");
                        }
                        break;
                    }
                }
            }
            this.writer.write('\n');
        }
        this.writer.write("END SGROUP\n");
    }

    private void writeMol(IAtomContainer mol) throws IOException, CDKException {
        this.writeHeader(mol);
        List<Sgroup> sgroups = this.getSgroups(mol);
        int numSgroups = 0;
        for (int i = 0; i < sgroups.size(); ++i) {
            if (sgroups.get(i).getType() == SgroupType.ExtMulticenter) continue;
            ++numSgroups;
        }
        this.writer.write("BEGIN CTAB\n");
        this.writer.write("COUNTS ").write(mol.getAtomCount()).write(' ').write(mol.getBondCount()).write(' ').write(numSgroups).write(" 0 0\n");
        HashMap<IChemObject, Integer> idxs = new HashMap<IChemObject, Integer>();
        HashMap<IAtom, ITetrahedralChirality> atomToStereo = new HashMap<IAtom, ITetrahedralChirality>();
        IAtom[] atoms = this.pushHydrogensToBack(mol, idxs);
        for (IBond bond : mol.bonds()) {
            idxs.put(bond, 1 + idxs.size() - mol.getAtomCount());
        }
        for (IStereoElement se : mol.stereoElements()) {
            if (!(se instanceof ITetrahedralChirality)) continue;
            atomToStereo.put(((ITetrahedralChirality)se).getChiralAtom(), (ITetrahedralChirality)se);
        }
        this.writeAtomBlock(mol, atoms, idxs, atomToStereo);
        this.writeBondBlock(mol, idxs);
        this.writeSgroupBlock(sgroups, idxs);
        this.writer.write("END CTAB\n");
        this.writer.writeDirect("M  END\n");
        this.writer.writer.flush();
    }

    @Override
    public void write(IChemObject object) throws CDKException {
        try {
            if (!IAtomContainer.class.isInstance(object)) {
                throw new CDKException("Unsupported ChemObject " + object.getClass());
            }
            this.writeMol((IAtomContainer)object);
        }
        catch (IOException ex) {
            throw new CDKException("Could not write V3000 format", ex);
        }
    }

    @Override
    public void setWriter(Writer writer) throws CDKException {
        this.writer = new V30LineWriter(writer);
    }

    @Override
    public void setWriter(OutputStream writer) throws CDKException {
        this.setWriter(new OutputStreamWriter(writer, StandardCharsets.UTF_8));
    }

    @Override
    public IResourceFormat getFormat() {
        return MDLV3000Format.getInstance();
    }

    @Override
    public boolean accepts(Class<? extends IChemObject> c) {
        return c.isInstance(IAtomContainer.class);
    }

    @Override
    public void close() throws IOException {
        if (this.writer != null) {
            this.writer.close();
        }
    }

    private void initIOSettings() {
        this.programNameOpt = (StringIOSetting)this.addSetting(new StringIOSetting("ProgramName", IOSetting.Importance.LOW, "Program name to write at the top of the molfile header, should be exactly 8 characters long", "CDK"));
    }

    public void customizeJob() {
        for (IOSetting setting : this.getSettings()) {
            this.fireIOSettingQuestion(setting);
        }
    }

    private static final class V30LineWriter
    implements Closeable {
        private final DecimalFormat decimalFmt = new DecimalFormat("#.#####", DecimalFormatSymbols.getInstance(Locale.ROOT));
        public static final String PREFIX = "M  V30 ";
        public static final int LIMIT = 78;
        private final Writer writer;
        private int currLength = 0;

        public V30LineWriter(Writer writer) {
            this.writer = writer instanceof BufferedWriter ? writer : new BufferedWriter(writer);
        }

        V30LineWriter writeDirect(String str) throws IOException {
            this.writer.write(str);
            return this;
        }

        V30LineWriter writeDirect(char c) throws IOException {
            this.writer.write(c);
            return this;
        }

        private void writeUnbroken(String str) throws IOException {
            this.newLineIfNeeded();
            this.writePrefixIfNeeded();
            int len = str.length();
            if (this.currLength + len < 78) {
                this.writer.write(str);
                this.currLength += len;
            } else {
                for (int i = 0; i < len; ++i) {
                    this.write(str.charAt(i));
                }
            }
        }

        private void newLineIfNeeded() throws IOException {
            if (this.currLength == 78) {
                this.writer.write(45);
                this.writer.write(10);
                this.currLength = 0;
            }
        }

        private void writePrefixIfNeeded() throws IOException {
            if (this.currLength == 0) {
                this.writer.write(PREFIX);
                this.currLength = PREFIX.length();
            }
        }

        V30LineWriter write(double num) throws IOException {
            return this.write(this.decimalFmt.format(num));
        }

        V30LineWriter write(int num) throws IOException {
            return this.write(Integer.toString(num));
        }

        V30LineWriter write(String str) throws IOException {
            int i = str.indexOf(10);
            if (i < 0) {
                this.writeUnbroken(str);
            } else if (i == str.length() - 1) {
                this.writeUnbroken(str);
                this.currLength = 0;
            } else {
                throw new UnsupportedOperationException();
            }
            return this;
        }

        V30LineWriter write(char c) throws IOException {
            if (c == '\n' && this.currLength == PREFIX.length()) {
                return this;
            }
            if (c != '\n') {
                this.newLineIfNeeded();
            }
            this.writePrefixIfNeeded();
            this.writer.write(c);
            ++this.currLength;
            if (c == '\n') {
                this.currLength = 0;
            }
            return this;
        }

        V30LineWriter write(Collection<? extends IChemObject> chemObjects, Map<IChemObject, Integer> idxs) throws IOException {
            this.write(chemObjects.size());
            ArrayList<Integer> integers = new ArrayList<Integer>();
            for (IChemObject iChemObject : chemObjects) {
                integers.add(idxs.get(iChemObject));
            }
            Collections.sort(integers);
            for (Integer n : integers) {
                this.write(' ').write(n);
            }
            return this;
        }

        @Override
        public void close() throws IOException {
            this.writer.close();
        }
    }
}

