% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cr_works.R
\name{cr_works}
\alias{cr_works}
\alias{cr_works_}
\title{Search CrossRef works (articles)}
\usage{
cr_works(
  dois = NULL,
  query = NULL,
  filter = NULL,
  offset = NULL,
  limit = NULL,
  sample = NULL,
  sort = NULL,
  order = NULL,
  facet = FALSE,
  cursor = NULL,
  cursor_max = 5000,
  .progress = "none",
  flq = NULL,
  select = NULL,
  async = FALSE,
  ...
)

cr_works_(
  dois = NULL,
  query = NULL,
  filter = NULL,
  offset = NULL,
  limit = NULL,
  sample = NULL,
  sort = NULL,
  order = NULL,
  facet = FALSE,
  cursor = NULL,
  cursor_max = 5000,
  .progress = "none",
  parse = FALSE,
  flq = NULL,
  select = NULL,
  async = FALSE,
  ...
)
}
\arguments{
\item{dois}{Search by a single DOI or many DOIs.  Note that using this
parameter at the same time as the \code{query}, \code{limit}, \code{select} or \code{flq}
parameter will result in an error.}

\item{query}{Query terms}

\item{filter}{Filter options. See examples for usage examples
and \code{\link{filters}} for what filters are available.
\code{filter} is available for use with \code{cr_works} and
other \code{crossref} family functions with \code{works=TRUE}}

\item{offset}{Number of record to start at. Minimum: 1. For
\code{\link{cr_works}}, and any function setting \code{works = TRUE},
the maximum offset value is 10000. For larger requests use \code{cursor}.}

\item{limit}{Number of results to return in the query. Not relavant when
searching with specific dois. Default: 20. Max: 1000}

\item{sample}{(integer) Number of random results to return. when you use
the sample parameter, the rows and offset parameters are ignored.
Ignored unless \code{works=TRUE}. Max: 100}

\item{sort}{Field to sort on. Acceptable set of fields to sort on:
\itemize{
\item \code{score} OR \code{relevance} - Sort by relevance score
\item \code{updated} - Sort by date of most recent change to metadata.
Currently the same as deposited.
\item \code{deposited} - Sort by time of most recent deposit
\item \code{indexed} - Sort by time of most recent index
\item \code{published} - Sort by publication date
\item \code{published-print} - Sort by print publication date
\item \code{published-online} - Sort by online publication date
\item \code{issued} - Sort by issued date (earliest known publication date)
\item \code{is-referenced-by-count} - Sort by number of times this DOI is
referenced by other Crossref DOIs
\item \code{references-count} - Sort by number of references included in
the references section of the document identified by this DOI
}}

\item{order}{(character) Sort order, one of 'asc' or 'desc'}

\item{facet}{(logical) Include facet results. Boolean or string with
field to facet on. Valid fields are *, affiliation, funder-name,
funder-doi, orcid, container-title, assertion, archive, update-type,
issn, published, source, type-name, publisher-name, license,
category-name, assertion-group. Default: \code{FALSE}}

\item{cursor}{(character) Cursor character string to do deep paging.
Default is None. Pass in '*' to start deep paging. Any combination of
query, filters and facets may be used with deep paging cursors.
While the \code{limit} parameter may be specified along with cursor,
offset and sample cannot be used with the cursor. See
https://github.com/CrossRef/rest-api-doc#deep-paging-with-cursors}

\item{cursor_max}{(integer) Max records to retrieve. Only used when
cursor param used. Because deep paging can result in continuous requests
until all are retrieved, use this parameter to set a maximum number of
records. Of course, if there are less records found than this value,
you will get only those found. When cursor pagination is being used
the \code{limit} parameter sets the chunk size per request.}

\item{.progress}{Show a \code{plyr}-style progress bar? Options are "none", "text",
"tk", "win", and "time".  See \code{\link[plyr]{create_progress_bar}} for details
of each. Only used when passing in multiple ids (e.g., multiple DOIs, DOI prefixes,
etc.), or when using the \code{cursor} param. When using the \code{cursor} param,
this argument only accept a boolean, either \code{TRUE} or \code{FALSE}; any
non-boolean is coerced to \code{FALSE}.}

\item{flq}{field queries. One or more field queries. Acceptable set of
field query parameters are:
\itemize{
\item \code{query.container-title}    - Query container-title aka.
publication name
\item \code{query.author} - Query author first and given names
\item \code{query.editor} - Query editor first and given names
\item \code{query.chair}    - Query chair first and given names
\item \code{query.translator} - Query translator first and given names
\item \code{query.contributor} - Query author, editor, chair and
translator first and given names
\item \code{query.bibliographic} - Query bibliographic information, useful
for citation lookup. Includes titles, authors, ISSNs and publication years
\item \code{query.affiliation} - Query contributor affiliations
}

Note: \code{query.title} has been removed - use \code{query.bibliographic}
as a replacement}

\item{select}{(character) One or more field to return (only those fields
are returned)}

\item{async}{(logical) use async HTTP requests. Default: \code{FALSE}}

\item{...}{Named parameters passed on to \code{\link[crul]{verb-GET}}}

\item{parse}{(logical) Whether to output json \code{FALSE} or parse to
list \code{TRUE}. Default: \code{FALSE}}
}
\description{
Search CrossRef works (articles)
}
\note{
See the "Rate limiting" seciton in \link{rcrossref} to get
into the "fast lane"
}
\section{Deep paging (using the cursor)}{

When using the cursor, a character string called \code{next-cursor} is
returned from Crossref that we use to do the next request, and so on. We
use a while loop to get number of results up to the value of
\code{cursor_max}. Since we are doing each request for you, you may not
need the \code{next-cursor} string, but if you do want it, you can get
to it by indexing into the result like \code{x$meta$next_cursor}

Note that you can pass in curl options when using cursor, via \code{"..."}
}

\section{Beware}{

The API will only work for CrossRef DOIs.
}

\section{Functions}{

\itemize{
\item \code{cr_works()} - Does data request and parses to data.frame for
easy downstream consumption
\item \code{cr_works_()} - Does data request, and gives back json (default)
or lists, with no attempt to parse to data.frame's
}
}

\section{Explanation of some data fields}{

\itemize{
\item score: a term frequency, inverse document frequency score that
comes from the Crossref Solr backend, based on bibliographic metadata
fields title, publication title, authors, ISSN, publisher, and
date of publication.
}
}

\examples{
\dontrun{
# Works funded by the NSF
cr_works(query="NSF")

# Works that include renear but not ontologies
cr_works(query="renear+-ontologies")

# Filter
cr_works(query="global state", filter=c(has_orcid=TRUE), limit=3)
# Filter by multiple fields
cr_works(filter=c(has_orcid=TRUE, from_pub_date='2004-04-04'))
# Only full text articles
cr_works(filter=c(has_full_text = TRUE))
# has affilitation data
cr_works(filter=c(has_affiliation = TRUE))
# has abstract
cr_works(filter=c(has_abstract = TRUE))
# has clinical trial number
cr_works(filter=c(has_clinical_trial_number = TRUE))

# Querying dois
cr_works(dois='10.1063/1.3593378')
cr_works('10.1371/journal.pone.0033693')
cr_works(dois='10.1007/12080.1874-1746')
cr_works(dois=c('10.1007/12080.1874-1746','10.1007/10452.1573-5125',
   '10.1111/(issn)1442-9993'))

# progress bar
cr_works(dois=c('10.1007/12080.1874-1746','10.1007/10452.1573-5125'),
   .progress="text")

# Include facetting in results
cr_works(query="NSF", facet=TRUE)
## Get facets only, by setting limit=0
cr_works(query="NSF", facet=TRUE, limit=0)
## you can also set facet to a query
cr_works(facet = "license:*", limit=0)

# Sort results
cr_works(query="ecology", sort='relevance', order="asc")
res <- cr_works(query="ecology", sort='score', order="asc")
res$data$score
cr_works(query="ecology", sort='published')
x=cr_works(query="ecology", sort='published-print')
x=cr_works(query="ecology", sort='published-online')

# Get a random number of results
cr_works(sample=1)
cr_works(sample=10)

# You can pass in dot separated fields to filter on specific fields
cr_works(filter=c(award.number='CBET-0756451',
   award.funder='10.13039/100000001'))

# Use the cursor for deep paging
cr_works(query="NSF", cursor = "*", cursor_max = 300, limit = 100)
cr_works(query="NSF", cursor = "*", cursor_max = 300, limit = 100,
   facet = TRUE)
## with optional progress bar
x <- cr_works(query="NSF", cursor = "*", cursor_max = 1200, limit = 200, 
  .progress = TRUE)

# Low level function - does no parsing to data.frame, get json or a list
cr_works_(query = "NSF")
cr_works_(query = "NSF", parse=TRUE)
cr_works_(query="NSF", cursor = "*", cursor_max = 300, limit = 100)
cr_works_(query="NSF", cursor = "*", cursor_max = 300, limit = 100,
   parse=TRUE)

# field queries
## query.author
res <- cr_works(query = "ecology", flq = c(query.author = 'Boettiger'))

## query.container-title
res <- cr_works(query = "ecology",
  flq = c(`query.container-title` = 'Ecology'))

## query.author and query.bibliographic
res <- cr_works(query = "ecology",
  flq = c(query.author = 'Smith', query.bibliographic = 'cell'))

# select only certain fields to return
res <- cr_works(query = "NSF", select = c('DOI', 'title'))
names(res$data)

# asyc
queries <- c("ecology", "science", "cellular", "birds", "European",
  "bears", "beets", "laughter", "hapiness", "funding")
res <- cr_works(query = queries, async = TRUE)
res_json <- cr_works_(query = queries, async = TRUE)
unname(vapply(res_json, class, ""))
jsonlite::fromJSON(res_json[[1]])

queries <- c("ecology", "science", "cellular")
res <- cr_works(query = queries, async = TRUE, verbose = TRUE)
res

# time
queries <- c("ecology", "science", "cellular", "birds", "European",
  "bears", "beets", "laughter", "hapiness", "funding")
system.time(cr_works(query = queries, async = TRUE))
system.time(lapply(queries, function(z) cr_works(query = z)))
}
}
\references{
https://github.com/CrossRef/rest-api-doc
}
\seealso{
Other crossref: 
\code{\link{cr_funders}()},
\code{\link{cr_journals}()},
\code{\link{cr_licenses}()},
\code{\link{cr_members}()},
\code{\link{cr_prefixes}()},
\code{\link{cr_types}()}
}
\concept{crossref}
