% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils_analysis.R
\name{adjust_pval}
\alias{adjust_pval}
\title{Adjust P-values in a \code{dana} object}
\usage{
adjust_pval(
  dana_obj,
  padj_by = c("all", "terms"),
  padj_method = c("BH", "bonferroni", "BY", "fdr", "hochberg", "holm", "hommel", "IHW",
    "storey"),
  padj_method_LRT = c("BH", "bonferroni", "BY", "fdr", "hochberg", "holm", "hommel",
    "IHW", "storey"),
  verbose = TRUE,
  ...
)
}
\arguments{
\item{dana_obj}{A \code{dana} class object returned by the \code{dana()} function.}

\item{padj_by}{Character string. Whether P-value adjustment should be done globally
across all coefficients (\code{"all"}) or separately for each coefficient term (\code{"terms"}).}

\item{padj_method}{Character vector of one or more methods for adjusting P-values from
coefficient tests. Defaults to \code{"BH"}.}

\item{padj_method_LRT}{Character vector of one or more methods for adjusting P-values from
LRT tests. Defaults to \code{"BH"}. P-values from LRT tests will always be adjusted
independently for each LRT term.}

\item{verbose}{Logical. Whether to print informative messages. Defaults to \code{TRUE}.}

\item{...}{Additional arguments passed to \code{IHW::ihw()} or \code{qvalue::qvalue()}.}
}
\value{
A modified \code{dana} object with new columns in the \verb{$fit} and \verb{$lrt} data frames
for each adjusted P-value method applied (e.g. \code{padj_BH}, \code{padj_storey_group}).
}
\description{
Applies multiple testing correction to P-values from differential analysis results
returned by the \code{dana()} function. Supports multiple adjustment methods and both
coefficient and likelihood ratio test (LRT) P-values.
}
\details{
Available adjustment methods include: \code{"BH"}, \code{"bonferroni"}, \code{"BY"}, \code{"fdr"},
\code{"hochberg"}, \code{"holm"}, \code{"hommel"}, \code{"IHW"}, and \code{"storey"}.
}
\examples{
set.seed(123)
mock_X <- matrix(rnorm(20 * 5), nrow = 20)
colnames(mock_X) <- paste0("feat_", seq_len(5))
rownames(mock_X) <- paste0("sample_", seq_len(20))

sample_data <- data.frame(
  sample_id = rownames(mock_X),
  group = factor(rep(c("A", "B"), each = 10)),
  time = factor(rep(c("T1", "T2"), times = 10)),
  subject_id = factor(rep(seq_len(10), each = 2)),
  stringsAsFactors = FALSE
)
rownames(sample_data) <- sample_data$sample_id

fit_df <- data.frame(
  feat_id = rep(colnames(mock_X), each = 2),
  Coefficient = rep(c("(Intercept)", "groupB"), 5),
  Estimate = rnorm(10),
  `Pr(>|t|)` = runif(10),
  stringsAsFactors = FALSE
)

# Mock dana object
dana_obj <- list(
  X = mock_X,
  sdata = sample_data,
  formula_rhs = ~ group,
  fit = fit_df,
  lrt = data.frame(),
  ranef = data.frame()
)
class(dana_obj) <- "dana"

# Add adjusted P-values
dana_obj <- dana_obj |>
  adjust_pval(dana_obj,
              padj_method = c("BH", "bonferroni"),
              padj_method_LRT = NULL,
              padj_by = "terms",
              verbose = FALSE)

}
\seealso{
\itemize{
\item \code{\link[=dana]{dana()}} for fitting differential analysis models on omics datasets.
\item \code{\link[IHW:ihw.default]{IHW::ihw()}} for inverted hypothesis weighting method details.
\item \code{\link[qvalue:qvalue]{qvalue::qvalue()}} for qvalue method details.
}
}
