% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rotate_2d.R
\name{rotate_2d}
\alias{rotate_2d}
\title{Rotate the values around an origin in 2 dimensions}
\usage{
rotate_2d(
  data,
  degrees,
  x_col = NULL,
  y_col = NULL,
  suffix = "_rotated",
  origin = NULL,
  origin_fn = NULL,
  keep_original = TRUE,
  degrees_col_name = ".degrees",
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{degrees}{Degrees to rotate values counterclockwise. In \code{[-360, 360]}.
Can be a \code{vector} with multiple degrees.}

\item{x_col}{Name of x column in \code{`data`}. If \code{NULL} and \code{`data`} is a \code{vector},
the index of \code{`data`} is used. If \code{`data`} is a \code{data.frame}, it must be specified.}

\item{y_col}{Name of y column in \code{`data`}. If \code{`data`} is a \code{data.frame}, it must be specified.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{origin}{Coordinates of the origin to rotate around.
A \code{vector} with 2 elements (i.e. origin_x, origin_y).
Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{degrees_col_name}{Name of new column with the degrees. If \code{NULL}, no column is added.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with seven new columns containing
the rotated x-,y- and z-values and the degrees, radiuses and origin coordinates.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

The values are rotated counterclockwise around a specified origin.

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and rotating around e.g. the centroid
of each group.
}
\details{
Applies the following rotation matrix:\tabular{lll}{
   [ \eqn{cos \theta} \tab , \eqn{ -sin \theta} \tab ] \cr
   [ \eqn{sin \theta} \tab , \eqn{ cos \theta} \tab ] \cr
}


That is:

\eqn{x' = x cos \theta - y sin \theta}

\eqn{y' = x sin \theta + y cos \theta}

Where \eqn{\theta} is the angle in radians.

As specified at \href{https://en.wikipedia.org/wiki/Rotation_matrix}{Wikipedia/Rotation_matrix}.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(ggplot2)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "Index" = 1:12,
  "A" = c(1:4, 9:12, 15:18),
  "G" = rep(1:3, each = 4)
)

# Rotate values around (0, 0)
rotate_2d(df, degrees = 45, x_col = "Index", y_col = "A", origin = c(0, 0))

# Rotate A around the centroid
df_rotated <- df \%>\%
  rotate_2d(
    x_col = "Index",
    y_col = "A",
    degrees = c(0, 120, 240),
    origin_fn = centroid
  )
df_rotated

# Plot A and A rotated around overall centroid
ggplot(df_rotated, aes(x = Index_rotated, y = A_rotated, color = factor(.degrees))) +
  geom_hline(yintercept = mean(df$A), size = 0.2, alpha = .4, linetype = "dashed") +
  geom_vline(xintercept = mean(df$Index), size = 0.2, alpha = .4, linetype = "dashed") +
  geom_line(alpha = .4) +
  geom_point() +
  theme_minimal() +
  labs(x = "Index", y = "Value", color = "Degrees")

# Rotate around group centroids
df_grouped <- df \%>\%
  dplyr::group_by(G) \%>\%
  rotate_2d(
    x_col = "Index",
    y_col = "A",
    degrees = c(0, 120, 240),
    origin_fn = centroid
  )
df_grouped

# Plot A and A rotated around group centroids
ggplot(df_grouped, aes(x = Index_rotated, y = A_rotated, color = factor(.degrees))) +
  geom_point() +
  theme_minimal() +
  labs(x = "Index", y = "Value", color = "Degrees")
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}

Other rotation functions: 
\code{\link{rotate_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{mutate functions}
\concept{rotation functions}
