% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distance.R
\name{distance}
\alias{distance}
\title{Calculate the distance to an origin}
\usage{
distance(
  data,
  cols = NULL,
  origin = NULL,
  origin_fn = NULL,
  distance_col_name = ".distance",
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{cols}{Names of columns in \code{`data`} to measure distance in.
Each column is considered a dimension.}

\item{origin}{Coordinates of the origin to calculate distances to.
A scalar to use in all dimensions
or a \code{vector} with one scalar per dimension.

\strong{N.B.} Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{distance_col_name}{Name of new column with the distances.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the additional columns (distances and origin coordinates).
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Calculates the distance to the specified origin with:
\deqn{d(P1, P2) = sqrt( (x2 - x1)^2 + (y2 - y1)^2 + (z2 - z1)^2 + ... )}

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and finding the distance to e.g. the centroid
of each group.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "x" = runif(20),
  "y" = runif(20),
  "g" = rep(1:4, each = 5)
)

# Calculate distances in the two dimensions (x and y)
# With the origin at x=0.5, y=0.5
distance(
  data = df,
  cols = c("x", "y"),
  origin = c(0.5, 0.5)
)

# Calculate distances to the centroid for each group in 'g'
distance(
  data = dplyr::group_by(df, g),
  cols = c("x", "y"),
  origin_fn = centroid
)
}
\seealso{
Other measuring functions: 
\code{\link{angle}()},
\code{\link{vector_length}()}

Other distance functions: 
\code{\link{closest_to}()},
\code{\link{dim_values}()},
\code{\link{expand_distances}()},
\code{\link{expand_distances_each}()},
\code{\link{furthest_from}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{distance functions}
\concept{measuring functions}
