#' @name exportFiles
#' @aliases exportFiles.redcapApiConnection
#' @aliases exportFiles.redcapDbConnection
#' @export exportFiles
#' @export exportFiles.redcapApiConnection
#' @export exportFiles.redcapDbConnection
#' @importFrom httr POST
#' 
#' @title Exports a File attached to a Record
#' @description A single file from a single record is retrieved.  The behavior 
#' of this function is consistent with the behavior of the API, which only 
#' allows one file to be downloaded at a time
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param record The record ID in which the desired file is stored. Must be length 1.
#' @param field The field name in which the file is stored. Must be length 1. 
#' @param event The event name for the file.  Must be length 1.  
#'   This applies only to longitudinal projects.  If the event is not
#'   supplied for a longitudinal project, the API will return an error message
#' @param dir A directory/folder to which the file will be saved. 
#'   By default, the working directory is used
#' @param filePrefix Logical.  Determines if a prefix is appended to the file 
#'   name.  The prefix takes the form [record_id]-[event_name]-[file_name].  
#'   The file name is always the same name of the file as it exists in REDCap
#' @param ... Arguments to be passed to other methods
#' @param proj A \code{redcapProject} object as created by \code{redcapProjectInfo}.
#' 
#' @details The function may only export a single file.  
#' See the examples for suggestions on exporting multiple files.
#' 
#' Note that the name of the file can not be changed.  Whatever name exists in 
#' REDCap is the name that will be used, although the record ID and event name 
#' may be appended as a prefix
#' 
#' @author Benjamin Nutter
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @examples
#'  \dontrun{
#'  > #*** Note: I cannot provide working examples without
#'  > #*** compromising security.  Instead, I will try to 
#'  > #*** offer up sample code with the matching results
#'  > 
#'  > #*** Create the connection object
#'  > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#'  > 
#'  >
#'  > #* Export a single file
#'  > exportFiles(rcon, record=1, field="file_upload", event="event_1_arm_1")
#'  The file was saved to '1-event_1_arm_1-NewOutcomes.xlsx'
#'  >
#'  >
#'  > #* Export all files in a project
#'  > #* Although this example only shows one field for files, it could work with
#'  > #* an arbitrary number of file upload fields
#'  > library(reshape2)
#'  > Data <- exportRecords(Data)
#'  > (filesToExport <- melt(Data[, c("id", "redcap_event_name", "file_upload")],
#'                           c("id", "redcap_event_name")),
#'                           na.rm=TRUE)
#'  id redcap_event_name    variable      value
#'  1  1     event_1_arm_1 file_upload [document]
#'  4  2     event_1_arm_1 file_upload [document]
#'  >
#'  > for(i in 1:nrow(filesToExport)){
#'  +   exportFiles(rcon, record=filesToExport$id[i],
#'  +               field=filesToExport$variable[i],
#'  +               event=filesToExport$redcap_event_name[i])
#'  + }
#'  The file was saved to '1-event_1_arm_1-NewOutcomes.xlsx'
#'  The file was saved to '2-event_1_arm_1-Sunset2.JPG'
#'  }
  
exportFiles <- function(rcon, record, field, event, dir, filePrefix=TRUE, ...,
                        proj=NULL)
  UseMethod("exportFiles")

#' @rdname exportFiles

exportFiles.redcapDbConnection <- function(rcon, record, field, event, dir, filePrefix=TRUE, ..., 
                        proj=NULL){
  message("Please accept my apologies.  The exportFiles method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname exportFiles

exportFiles.redcapApiConnection <- function(rcon, record, field, event, dir, filePrefix=TRUE, ...,
                        proj=NULL){
  #* Use working directory if 'dir' is not specified
  if (missing(dir)) dir <- getwd()
  
  #* stop the function if arguments do not specify a unique record-event
  if (missing(event)) event <- ""
  if (any(sapply(list(record, field, event), length) > 1)){
    stop("The arguments 'record', 'field', and 'event' may each only have length 1")
  }
  
  #* make sure 'field' exist in the project and are 'file' fields
  if (is.null(proj$meta_data)) meta_data <- exportMetaData(rcon)
  if (!field %in% meta_data$field_name) stop(paste("'", field, "' does not exist in the project.", sep=""))
  if (meta_data$field_type[meta_data$field_name == field] != "file")
      stop(paste0("'", field, "' is not of field type 'file'"))
      
  #* make sure 'event' exists in the project
  if (is.null(proj$events)) events_list <- exportEvents(rcon)
  if (class(events_list) == 'data.frame'){
    if (!event %in% events_list$unique_event_name) 
      stop(paste0("'", event, "' is not a valid event name in this project."))
  }

  .params <- list(token=rcon$token, content='file',
                  action='export', returnFormat='csv',
                  record=record,
                  field=field)
  if (event != "") .params[['event']] <- event
  
  #* Export the file
  x <- httr::POST(url=rcon$url, body=.params, config=rcon$config)
  if (x$status_code == 200){
    #* strip the returned character string to just the file name.
    filename = sub("[[:print:]]+; name=", "", x$headers$'content-type')
    filename = gsub("\"", "", filename)
    filename <- sub(";charset[[:print:]]+", "", filename)
    
    #* Add the prefix
    if (filePrefix) filename <- paste(record, "-", event, "-", filename, sep="")
    
    #* Write to a file
    writeBin(as.vector(x$content), file.path(dir, filename), 
             useBytes=TRUE)
    message(paste("The file was saved to '", filename, "'", sep=""))
  }
  else{                 
   stop(paste0(x$status_code, ": ", as.character(x)))
  }
}
