#' @name deleteFiles
#' @aliases deleteFiles.redcapApiConnection
#' @aliases deleteFiles.redcapDbConnection
#' @export deleteFiles
#' @importFrom httr POST
#' 
#' @title Delete a File attached to a Record
#' @description This function allows you to remove a document that has been 
#' attached to an individual record
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param record The record ID in which the desired file is stored. Must be length 1.
#' @param field The field name in which the file is stored. Must be length 1.
#' @param event The event name for the file.  Must be length 1.  
#'   This applies only to longitudinal projects.  If the event is not
#'   supplied for a longitudinal project, the API will return an error message.
#' @param proj A \code{redcapProject} object as created by \code{redcapProjectInfo}.
#' @param ... Arguments to be passed to other methods
#' 
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#'  
#' @examples
#' \dontrun{
#'   > #*** Note: I cannot provide working examples without
#'   > #*** compromising security.  Instead, I will try to 
#'   > #*** offer up sample code with the matching results
#'   > 
#'   > #*** Create the connection object
#'   > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#'   > 
#'   >
#'   > #* Delete a file
#'   > deleteFiles(rcon, record=1, field="file_upload", event="event_1_arm_1")
#'   The file was successfully deleted
#'   >
#'   }
#'   
  
deleteFiles <- function(rcon, record, field, event, ...)
  UseMethod("deleteFiles")

#' @rdname deleteFiles
#' @export 
deleteFiles.redcapDbConnection <- function(rcon, record, field, event, ...){
  message("Please accept my apologies.  The deleteFiles method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname deleteFiles
#' @export

deleteFiles.redcapApiConnection <- function(rcon, record, field, event, ..., 
                                            proj=NULL){
  #* stop the function if arguments do not specify a unique record-event
  if (missing(event)) event <- ""
  if (any(sapply(list(record, field, event), length) > 1)){
    stop("The arguments 'record', 'field', and 'event' may each only have length 1")
  }
  
  #* make sure 'field' exist in the project and are 'file' fields
  if (is.null(proj$meta_data)) meta_data <- exportMetaData(rcon, config=rcon$config)
  if (!field %in% meta_data$field_name) stop(paste0("'", field, "' does not exist in the project."))
  if (meta_data$field_type[meta_data$field_name == field] != "file")
    stop(paste0("'", field, "' is not of field type 'file'"))
  
  #* make sure 'event' exists in the project
  if (is.null(proj$events)) events_list <- exportEvents(rcon)
  if (class(events_list) == "data.frame"){
    if (!event %in% events_list$unique_event_name) 
      stop(paste0("'", event, "' is not a valid event name in this project."))
  }
  
  .params <- list(token=rcon$token, content='file',
                  action='delete', record=record,
                  field=field, returnFormat='csv')
  if (event != "") .params[['event']] <- event
  
  #* Delete the file
  #* The tryCatch here seems a little quirky.  My best understanding is that since the API isn't returning
  #* anything into the 'content' attribute returned by POST, POST is casting an error.  Oddly, an error in this
  #* case, an error means the action was successfully performed.  The tryCatch call negotiates that oddity to
  #* get the desired result.
  x <- tryCatch(httr::POST(url=rcon$url, body=.params, config=rcon$config),
                error=function(cond) list(status_code=200))
  if (x$status_code != "200")stop(paste(x$status_code, ": ", as.character(x), sep=""))
  else message("The file was successfully deleted")
  
}

