% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ref.MCMC.R
\name{ref.MCMC}
\alias{ref.MCMC}
\title{MCMC for Reference Prior on an Intrinsic Conditional Autoregressive Random Effects Model for Areal Data}
\usage{
ref.MCMC(y, X, H, iters = 10000, burnin = 5000, verbose = TRUE,
  tauc.start = 1, beta.start = 1, sigma2.start = 1,
  step.tauc = 0.5, step.sigma2 = 0.5)
}
\arguments{
\item{y}{Vector of responses.}

\item{X}{Matrix of covariates.  This should include a column of 1's for models with a non-zero intercept.}

\item{H}{The neighborhood matrix.}

\item{iters}{Number of MCMC iterations to perform.  Defaults to 10,000.}

\item{burnin}{Number of MCMC iterations to discard as burn-in.  Defaults to 5,000.}

\item{verbose}{If FALSE, MCMC progress is not printed.}

\item{tauc.start}{Starting value for the spatial dependence parameter.}

\item{beta.start}{Starting value for the vector of fixed effect regression coefficients.}

\item{sigma2.start}{Starting value for the variance of the unstructured random effects.}

\item{step.tauc}{Step size for the spatial dependence parameter}

\item{step.sigma2}{Step size for the variance of the unstructured random effects.}
}
\value{
A list containing MCMC chains and parameter summaries.
    \item{MCMCchain}{Matrix of MCMC chains.}
    \item{tauc.MCMC}{MCMC chains for the spatial dependence parameter.}
    \item{sigma2.MCMC}{MCMC chains for the variance of the unstructured random effects.}
    \item{phi.MCMC}{MCMC chains for the spatial random effects.}
    \item{beta.MCMC}{MCMC chains for the fixed effect regression coefficients.}
    \item{accept.sigma2}{Final acceptance number for variance of the unstructured random effects.}
    \item{accept.tauc}{Final acceptance number for spatial dependence parameter.}
    \item{accept.phi}{Final acceptance number for spatial random effects.}
}
\description{
Implements the Metropolis-within-Gibbs sampling algorithm proposed by Keefe et al. (2018),
to perform posterior inference for the intrinsic conditional autoregressive model with spatial
random effects.
}
\examples{
#### Fit the model for simulated areal data on a grid ####

### Load extra libraries
library(sp)
library(methods)
library(spdep)
library(mvtnorm)

### Generate areal data on a grid
rows=5; cols=5
tauc=1
sigma2=2; beta=c(1,5)

### Create grid
grid <- GridTopology(c(1,1), c(1,1), c(cols,rows))
polys <- as(grid, "SpatialPolygons")
spgrid <- SpatialPolygonsDataFrame(polys,data=data.frame(row.names=row.names(polys)))

### Create neighborhood matrix
grid.nb <- poly2nb(spgrid,queen=FALSE)
W <- nb2mat(grid.nb, style="B")

### Put spatially correlated data in grid
p <- length(beta)
num.reg <- (rows*cols)
if(p>1){x1<-rmvnorm(n=num.reg,mean=rep(0,p-1),sigma=diag(p-1))} else{x1<-NULL}
X <- cbind(rep(1,num.reg),x1)
Dmat <- diag(apply(W,1,sum))
H <- Dmat - W
row.names(H) <- NULL

### Obtain true response vector
theta_true <- rnorm(num.reg,mean=0,sd=sqrt(sigma2))
Q <- eigen(H,symmetric=TRUE)$vectors
eigH <- eigen(H,symmetric=TRUE)$values
D <- diag(eigH)
Qmat <- Q[,1:(num.reg-1)]
phimat <- diag(1/sqrt(eigH[1:(num.reg-1)]))
z <- t(rmvnorm(1,mean=rep(0,num.reg-1),sigma=diag(num.reg-1)))
phi_true <- sqrt((1/tauc)*sigma2)*(Qmat\%*\%phimat\%*\%z)
Y <- X\%*\%beta + theta_true + phi_true

### Fit the model
set.seed(5432)
\donttest{model <- ref.MCMC(y=Y,X=X,H=H,iters=15000,burnin=5000,verbose=TRUE,tauc.start=.1,beta.start=-1,
sigma2.start=.1,step.tauc=0.5,
      step.sigma2=0.5)
      }

#### Small example for checking
model <- ref.MCMC(y=Y,X=X,H=H,iters=1000,burnin=50,verbose=TRUE,tauc.start=.1,beta.start=-1,
sigma2.start=.1,step.tauc=0.5,
      step.sigma2=0.5)
}
\references{
\insertRef{keefe2018}{ref.ICAR}

\insertRef{Keefe_2018}{ref.ICAR}
}
\author{
Erica M. Porter, Matthew J. Keefe, Christopher T. Franck, and Marco A.R. Ferreira
}
