% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relMix.R
\name{relMix}
\alias{relMix}
\title{Relationship inference based on mixtures}
\usage{
relMix(
  pedigrees,
  locus,
  R,
  datamatrix,
  ids,
  D = rep(list(c(0, 0)), length(ids)),
  di = 0,
  kinship = 0
)
}
\arguments{
\item{pedigrees}{A list of pedigrees defined using \code{\link{FamiliasPedigree}}}

\item{locus}{A list of class \code{\link{FamiliasLocus}} containing information about the locus. Note that a silent allele must be indicated by 's' (and not 'silent' as in Familias)}

\item{R}{A vector of mixture alleles, or a list of such if there are multiple replicates}

\item{datamatrix}{A data frame where each line corresponds to one constellation of genotypes for the involved individuals.
Indices of individuals must be given as rownames and must correspond to indices in the pedigree}

\item{ids}{Index vector indicating which individuals are contributors to the mixture. The indices must correspond to
indices in the pedigree}

\item{D}{List of numeric dropout values (between 0 and 1) per contributor.
Each element is a vector containing heterozygous and homozygous dropout probability for the given contributor}

\item{di}{A numeric drop-in value (between 0 and 1)}

\item{kinship}{A numeric value between 0 and 1 that defines the theta-parameter}
}
\value{
A numeric likelihood for each pedigree named according to the pedigrees, and a matrix of likelihoods for each pedigree and each term (genotype constellation) considered in the calculation
(one row per term).
}
\description{
Calculates likelihoods for relationship inference involving mixtures and missing reference profiles,
including drop-in and dropout, mutations, silent alleles and theta correction.
}
\details{
The function requires the package \code{\link{Familias}} and calls on the function \code{\link{FamiliasPosterior}}.
}
\examples{
#Example 1: paternity trio with mixture of mother and child
#Define alleles and frequencies
alleles <- 1:2
afreq <- c(0.4,0.6)
#Define pedigrees
persons <- c("CH","MO","AF")
ped1 <- Familias::FamiliasPedigree(id=persons, dadid=c("AF",NA, NA), momid=c("MO", NA,NA),
                        sex=c("male", "female", "male"))
ped2 <- Familias::FamiliasPedigree(id=c(persons, "TF"), dadid=c("TF", NA, NA,NA),
                        momid=c("MO", NA, NA,NA), sex=c("male", "female", "male", "male"))
pedigrees <- list(isFather = ped1, unrelated=ped2)
#Create locus object
locus <- Familias::FamiliasLocus(frequencies=afreq,name="M1",
                      allelenames= alleles)
#Known genotypes of alleged father and mother
gAF <- c(1,1)
gMO <- c(1,1)
#Mixture alleles
R <- c(1,2)
datamatrix <- createDatamatrix(locus,knownGenos=list(AF=gAF,MO=gMO),idsU=c("CH"))
#Define dropout and drop-in values
d <- 0.1
di <- 0.05
res <- relMix(pedigrees, locus, R, datamatrix, ids=c("MO","CH"),
             D=list(c(0,0),c(d,d^2)),di=di, kinship=0)
#LR=0.054
res$isFather/res$unrelated

#Example 2: Exhaustive example with silent allele, mutations, dropout and drop-in
#H1: Contributors are mother and child
#H2: Contributors are mother and unrelated
#Possible dropout in both contributors
gMO <- c(1,1) #Mother's genotype
R <- 1 #Mixture alleles
#Mother/child pedigree
persons <- c("CH","MO")
ped1 <- Familias::FamiliasPedigree(id=persons, dadid=c(NA,NA), momid=c("MO", NA),
                        sex=c("male", "female"))
ped2 <- Familias::FamiliasPedigree(id=c(persons), dadid=c(NA, NA),
                        momid=c( NA, NA),
                        sex=c("male", "female"))
pedigrees <- list(H1 = ped1, H2=ped2)
#Alleles and frequencies:
#When silent alleles are involved, a custom mutation matrix is required.
#No mutations are possible to or from silent alleles.
#We create the mutation model with FamiliasLocus and modify it before it is
#passed on to relMix
alleles <- c(1,2,'silent')
afreq <- c(0.4,0.5,0.1)
#Create initial locus object with mutation matrix
locus <- Familias::FamiliasLocus(frequencies=afreq,name='M1',
                      allelenames= alleles, MutationModel='Equal',
                      femaleMutationRate=0.1,maleMutationRate=0.1)
#Modify mutation matrix from Familias:
#Silent allele must be given as 's' (not 'silent' as in Familias)
newAlleles <- c(alleles[-length(alleles)],'s')
mm <- locus$femaleMutationMatrix
colnames(mm) <- rownames(mm) <- newAlleles
#Create new locus object with modified mutation matrix
locus <- Familias::FamiliasLocus(frequencies=afreq,name='M1',
                      allelenames= newAlleles, MutationModel='Custom', MutationMatrix=mm)
knownGenos <- list(gMO)
names(knownGenos) <- c("MO")
datamatrix <- createDatamatrix(locus,knownGenos,ids="CH")
d <- 0.1 #Dropout probability for both contributors
di <- 0.05
res2 <- relMix(pedigrees, locus, R, datamatrix, ids=c("MO","CH"),
              D=list(c(d,d^2),c(d,d^2)),di=di, kinship=0)
#LR=1.68
res2$H1/res2$H2

}
\references{
Dorum et al. (2017) <doi:10.1007/s00414-016-1526-x> \cr
Kaur et al. (2016) <doi:10.1007/s00414-015-1276-1> \cr
}
\seealso{
\code{\link{relMixGUI}} for the GUI version of relMix, \code{\link{FamiliasLocus}} on how to create a FamiliasLocus and \code{\link{FamiliasPedigree}} on how to create a FamiliasPedigree.
}
\author{
Navreet Kaur, Thore Egeland, Guro Dorum
}
