\docType{package}
\name{reutils}
\alias{reutils}
\alias{reutils-package}
\title{An interface to the NCBI Entrez programming Utilities.}
\description{
  \code{reutils} provides support for interacting with NCBI
  databases such as PubMed, Genbank, or GEO via the Entrez
  Programming Utilities
  (\href{http://www.ncbi.nlm.nih.gov/books/NBK25501/}{EUtils}).

  Please check the relevant
  \href{http://www.ncbi.nlm.nih.gov/books/NBK25497/\#chapter2.Usage_Guidelines_and_Requiremen}{usage
  guidelines} when using these services. Note that Entrez
  server requests are subject to frequency limits.
}
\details{
  With nine E-Utilities, NCBI provides a programmatical
  interface to the Entrez query and database system for
  searching and retrieving requested data

  Each of these tools corresponds to an \code{R} function
  in the \code{reutils} package described below.

  The output returned by the EUtils is typically in XML
  format. To gain access to this output you have several
  options:

  \enumerate{ \item Use the \code{content(as = "xml")}
  method to extract the output as an
  \code{XMLInternalDocument} object and process it further
  using the facilities provided by the \code{XML} package.

  \item Use the \code{content(as = "parsed")} method to
  extract the output into \code{data.frame}s. Note that
  this is currently only implemented for \emph{docsum}s
  returned by \code{\link{esummary}}, \emph{uilist}s
  returned by \code{\link{esearch}}, and the output
  returned by \code{\link{einfo}}.

  \item Access specific nodes in the XML tree using XPath
  expressions with the reference class methods
  \code{#xmlValue}, \code{#xmlAttr}, or \code{#xmlName}
  built into \code{\link{eutil}} objects. }

  The Entrez Programming Utilities can also generate output
  in other formats, such as plain-text Fasta or GenBank
  files for sequence databases, or the MedLine format for
  the literature database. The type of output is generally
  controlled by setting the \code{retmode} and
  \code{rettype} arguments when calling a EUtil.
}
\section{Main functions}{
  \itemize{ \item \code{\link{esearch}}: Search and
  retrieve primary UIDs for use with \code{esummary},
  \code{elink}, or \code{efetch}.  \code{esearch}
  additionally returns term translations and optionally
  stores results for future use in the user's Web
  Environment.

  \item \code{\link{esummary}}: Retrieve document summaries
  from a list of primary UIDs (Provided as a character
  vector or as an \code{esearch} object).

  \item \code{\link{egquery}}: Provides Entrez database
  counts in XML for a single search term using a Global
  Query.

  \item \code{\link{einfo}}: Retrieve field names, term
  counts, last update, and available updates for each
  database.

  \item \code{\link{efetch}}: Retrieve data records in a
  specified format corresponding to a list of primary UIDs
  or from the user's Web Environment in the Entrez History
  server.

  \item \code{\link{elink}}: Returns a list of UIDs (and
  relevancy scores) from a target database that are related
  to a list of UIDs in the same database or in another
  Entrez database.

  \item \code{\link{epost}}: Uploads primary UIDs to the
  users's Web Environment on the Entrez history server for
  subsequent use with \code{esummary}, \code{elink}, or
  \code{efetch}.

  \item \code{\link{espell}}: Provide spelling suggestions.

  \item \code{\link{ecitmatch}}: Retrieves PubMed IDs
  (PMIDs) that correspond to a set of input citation
  strings

  \item \code{\link{content}}: Extract the content of a
  request from the \code{\link{eutil}} object returned by
  any of the above functions. }
}

\section{Package options}{
  \emph{reutils} uses two \code{\link{options}} to
  configure behaviour:

  \itemize{ \item \code{reutils.email}: NCBI requires that
  a user of their API provides an email address with a call
  to Entrez. If you are going to perform a lot of queries
  you should set \code{reutils.email} to your email address
  in your .Rprofile file. \emph{reutils} will annoy you
  with warnings until you comply.

  \item \code{reutils.verbose.queries}: If you perform many
  queries interactively you might want to get messages
  announcing the queris you run. You can do so by setting
  the option \code{reutils.verbose.queries} to \code{TRUE}.

  }
}
\examples{
############################################################################
#
# combine esearch and efetch
#
# Download PubMed records that are indexed in MeSH for both 'Clamydia' and 
# 'genome' and were published in 2013.
query <- "Chlamydia[mesh] and genome[mesh] and 2013[pdat]"

# Upload the PMIDs for this search to the History server
pmids <- esearch(query, "pubmed", usehistory=TRUE)
pmids

# Fetch the records
articles <- efetch(pmids)

# Use XPath expressions with the #xmlValue() or #xmlAttr() methods to directly
# extract specific data from the XML records stored in the 'efetch' object.
titles <- articles$xmlValue("//ArticleTitle")
abstracts <- articles$xmlValue("//AbstractText")


############################################################################
#
# combine epost with esummary/efetch
#
# Download protein records corresponding to a list of GI numbers.
uid <- c("194680922", "50978626", "28558982", "9507199", "6678417")

# post the GI numbers to the Entrez history server
p <- epost(uid, "protein")

# retrieve docsums with esummary
docsum <- content(esummary(p, version="1.0"), "parsed")
docsum

# download FASTAs as 'text' with efetch
prot <- efetch(p, retmode="text", rettype="fasta")
prot

# retrieve the content from the efetch object
fasta <- content(prot)
}
\author{
  Gerhard Schoefl \email{gschofl@yahoo.de}
}
\keyword{package}

