#' Get Hold Details for An Account Using Currency
#'
#' @name holds
#'
#' @description This is an auth based function. User must have valid api keys generated by GADX which must be passed as mandatory arguments. The function takes a currency as an additional input and returns the hold information. Since currency and account id have one to one mapping, the currency is being used a proxy. The basic hold amounts can also be fetched using \code{\link{account}} or \code{\link{accounts}}, however this function provides additional details and ties to active open orders. Please note that each trade buy order, results in a hold on the currency like \code{USD} and each sell order will result in a hold on the cryptoasset like \code{BTC}.
#'
#' @param currency Optional character value. The default is \code{"LTC"}. This is case insensitive and must be a valid currency as provided by \code{\link{accounts}} or \code{\link{account}}.
#' @param api.key Mandatory character value. This is the API key as generated by GDAX. Typically a 32 character value.
#' @param secret Mandatory character value. This is the API secret as generated by GDAX. Typically a 88 character value.
#' @param passphrase Mandatory character value. This is the passphrase as generated by GDAX. Typically a 11 character value.
#'
#' @return  Dataframe with timestamp of all the holds for the currency, the amount & type of hold and a reference id which created the hold.
#'
#' @examples
#' \dontrun{
#' holds(api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' holds("ETH", api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' holds("USD", api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' }
#'
#' @export

holds <- function(currency = "LTC",
                  api.key,
                  secret,
                  passphrase) {
  #Determine account id----
  raw_data <- accounts(api.key, secret, passphrase)

  #case remediation----
  currency <- toupper(currency)

  #valid currency?----
  if (currency %in% raw_data$currency) {
    acct_id <- raw_data$id[raw_data$currency == currency]

    #get url extension----
    req.url <-  paste0("/accounts/", acct_id, "/holds")

    #define method----
    method <- "GET"

    #fetch response----
    response <- auth(
      method = method,
      req.url = req.url,
      api.key = api.key,
      secret = secret,
      passphrase = passphrase
    )

    #transform----
    response <- as.data.frame(response)

    #return----
    return(response)
  }
  #Invalid currency?----
  else {
    stop("Invalid currency provided")
  }
}
