ringscale.batch <- function(img, ring.masks) {
	# ringscale an image 'img'
	
	img.dim <- dim(img)
	X <- img.dim[1]
	Y <- img.dim[2]
	
	# how many rings are there?
	rings <- dim(ring.masks)[3] - 1
	
	# for each ring...
	for (i in 1:rings) {
		# ...get its mask...
		mask <- ring.masks[,,i]
		
		# ...apply it on the image...
		img.scaled.return <- img.scale(img[mask])
		
		# ...and scale the current ring of the image
		img.scaled <- img.scaled.return$"image"
		img.scaled.min <- img.scaled.return$"lower"
		img.scaled.max <- img.scaled.return$"upper"
		img[mask] <- (img.scaled - img.scaled.min) / (img.scaled.max - img.scaled.min)
	}
	
	# ringscale only applies to a circular region, so set
	# the remains (the four cournes, that is) 'NA'
	is.na(img[ring.masks[,,rings+1]]) <- TRUE
	
	img
}

roi.candidates <- function(img, ring.masks, D.speckle, theta = 0.8) {
	# creates a candidate list from a ROI image 'img' by first
	# applying ringscale and then compiling the hypotheses
	
	img.dim <- dim(img)
	X <- img.dim[1]
	Y <- img.dim[2]
	
	img.max <- max(img)
	
	img.min <- min(img, na.rm = TRUE)
	img.work <- img
	
	img.work <- ringscale.batch(img.work, ring.masks = ring.masks)
	
	# we apply a lowpass filter, so we set all NAs to 0
	# before doing that
	img.work[is.na(img.work)] <- 0
	img.work <- img.lowpass(img.work)
	img.work <- abs(img.work)
	threshold <- theta * max(img.work, na.rm = TRUE)
	
	candidates <- NULL
	
	repeat {
		img.work.max <- max(img.work, na.rm = TRUE)
		
		# stop, if we have fallen below the threshold
		if (img.work.max < threshold || img.work.max == -Inf)
			break;
		
		# add candidate's position to the list
		pos <- which(img.work == img.work.max, arr.ind = TRUE)[1,]
		candidates <- append(candidates, pos)
		
		# cut out found companion in the image
		mask <- t(mask.circle(width = X, height = Y, x0 = pos[1], y0 = pos[2], radius.outer = floor(D.speckle/2)))
		img.work[mask] <- NA
	}
	
	if (!is.null(candidates)) {
		candidates <- matrix(candidates, ncol = 2, byrow = TRUE)
		colnames(candidates) <- (c("x", "y"))
	}
	
	candidates
}

evaluate <- function(filename = character(0), support.min = 0.5) {
	# evaluates the result of generated by analysis();
	# i.e. shows the list of "final" hypotheses with their
	# ranks and support values

	# open file dialog
	if (length(filename) == 0) {
		if (graphical()) {
			library(tcltk)
			filename <- tclvalue(tkgetOpenFile(title = "Select Ringscale result file to evaluate", filetypes = "{{All files} *} {{Unfinished Ringscale session} {.ringscale.part}}", multiple = FALSE))
		}
	}
	
	if (!file.exists(filename)) {
		stop(paste("File ", filename, " does not exists.", sep = ""))
	}
	
	load(filename)
	
	# normalize the "votes" on [0,1] and set NAs to 0
	hypotheses <- hypotheses.combined$data / hypotheses.combined$info$image.count
	hypotheses[is.na(hypotheses)] <- 0
	
	hypotheses.dim <- dim(hypotheses)
	X <- hypotheses.dim[1]
	Y <- hypotheses.dim[2]
	
	# mask radius for the calculation of the support value
	mask.radius <- floor(hypotheses.combined$instrument$D.speckle/4)
	mask.diameter <- 2*mask.radius - 1
	mask <- mask.circle(width = mask.diameter, height = mask.diameter, x0 = mask.radius + 1, y0 = mask.radius + 1, radius.outer = mask.radius + 1)
	
	# show image of combined hypotheses
	img.astro(hypotheses.combined$data, pixscale = hypotheses.combined$instrument$pixscale, distance.pc = hypotheses.combined$object$distance.pc, name = hypotheses.combined$object$name, colormap = "heat")
	
	# skeleton of candidate list
	candidates <- matrix(data = NA, ncol = 7, nrow = 0)
	
	rank <- 0
	repeat {
		rank <- rank + 1
		
		# calculate the support values
		hypotheses.sum <- filter.FFT(hypotheses, mask)
		
		# if the biggest support value found is smaller than an
		# threshold, stop
		max.support <- max(hypotheses.sum, na.rm = TRUE)
		if (max.support < support.min)
			break
		
		# find the maximum support value
		max.pos <- which(hypotheses.sum == max.support, arr.ind = TRUE)[1,]
		
		# delete found hypothesis in the image and put it into the candidate list
		hypotheses <- hypotheses * !mask.circle(width = X, height = Y, x0 = max.pos["col"], y0 = max.pos["row"], radius.outer = mask.radius + 1)
		dist.px <- sqrt(sum((max.pos - (c(X,Y)+1)/2)^2))
		dist.arcsec <- dist.px * 1000 * hypotheses.combined$instrument$pixscale
		dist.AU <- dist.px * hypotheses.combined$instrument$pixscale * hypotheses.combined$object$distance.pc
		candidates <- matrix(byrow = TRUE, ncol = 7, data = append(t(candidates), c(rank, max.support, max.pos["col"] - (X+1)/2, max.pos["row"] - (Y+1)/2, dist.px, dist.arcsec, dist.AU)))
		
		# mark hypotheses in the image with a circle;
		# the linewidth represents the support value,
		# the radius of the circle represents the rank
		circle.add(pos.rel = (max.pos)/c(X,Y), radius.rel = 0.05*(0.5+1/2^rank), line.width = 10*max.support)
	}
	
	# return candidate list
	colnames(candidates) <- c("rank", "support", "Delta x", "Delta y", "dist [px]", "dist [mas]", "dist [AU]")
	
	candidates
}

analyze <- function (filenames = character(0), result.filename = paste(format(Sys.time(), "%Y-%m-%d-%H%M%S"), ".RData", sep = ""), evaluate = TRUE) {
	# analyzes all FITS files from a directory with the ringscale method,
	# this is the central function of this package;
	# if 'evaluate' is true, then the evaluation function 
	# is called after the analysis finished
	
	# we will have to read in some FITS files
	require(FITSio)

	# if no files are given, let the user select them
	if (length(filenames) == 0)
		if (graphical()) {
			library(tcltk)
			filenames <- strsplit(tclvalue(tkgetOpenFile(title = "Select FITS files to analyze", filetypes = "{{FITS file} {.fits}} {{Unfinished Ringscale session} {.ringscale.part}} {{All files} *}", multiple = TRUE)), split = " ")[[1]]
		}
	
	# stop, if no files were specified
	# this may also happen after an "empty" tcltk selection
	if (length(filenames) == 0)
		stop("No input files specified.")
	
	# check if the file is an unfinished session
	if (length(grep("\\.ringscale\\.part$", filenames)) > 0) {
		if (length(filenames) == 1) {
			# continue specified session
			cat("Resuming session ", filenames[1], "\n", sep = "")
			load(filenames[1])
			time.start <- hypotheses.combined$info$time.start
			image.count <- hypotheses.combined$info$image.count
			file.count <- hypotheses.combined$info$file.count
			part.continued <- TRUE
		} else {
			stop("Only can resume one session at a time.")
		}
	} else {
		part.continued <- FALSE
	}
	
	# if we do not continue a previous sessions, set some basic values
	if (!part.continued) {
		time.start <- Sys.time()
		
		# count processed images
		image.count <- 0
		
		# file list
		files <- rep(FALSE, length(filenames))
		names(files) <- filenames
		
		# this is the skeleton of the resulting data set
		hypotheses.combined <- list(info = list(time.start = time.start, time.last = Sys.time(), time.elapsed = 0, image.count = image.count, "files" = files, result.filename = result.filename, radius = 1), instrument = list(D = NA, lambda = NA, pixscale = NA, r0 = NA, D.speckle = NA), object = list(name = "UNKNOWN", distance.pc = NA), data = NULL)
		
		# read in FITS header to check for telescope values
		cat("Checking FITS header for needed values...\n")
		header <- readFITS(filenames[1])$hdr
		
		# read out important values (D, lambda, pixelscale, r0)
		# ask the user if values cannot be found:
		
		# telescope diameter
		if (header.find(header, "TELESCOP") == "ESO-VLT-U4") {
			hypotheses.combined$instrument$D <- 8.2
			cat("D =", hypotheses.combined$instrument$D, "m\n")
		} else
			if (interactive()) {
				repeat {
					input <- as.numeric(readline("Could not determine telescope diameter, please enter. D [m] = "))
					if (!is.na(input)) {
						hypotheses.combined$instrument$D <- input
						break
					}
				}
			}
		
		# pixel scale
		hypotheses.combined$instrument$pixscale <- instrument.get(header, "HIERARCHESOINSPIXSCALE", "the pixel scale", "pixscale", "arcsec/px")
		
		# observation wavelength
		hypotheses.combined$instrument$lambda <- instrument.get(header, "HIERARCHESOINSCWLEN", "the observation wavelength", "lambda", "micron")
		
		# Fried's parameter
		hypotheses.combined$instrument$r0 <- instrument.get(header, "HIERARCHESOAOSRTCDETDSTR0MEAN", "Fried's parameter", "r0", "cm")
		
		# calculate and print speckle diameter base on these values
		hypotheses.combined$instrument$D.speckle <- speckle.Diameter(hypotheses.combined$instrument$D, hypotheses.combined$instrument$lambda, hypotheses.combined$instrument$pixscale)
		cat("D.speckle = ", hypotheses.combined$instrument$D.speckle, " px\n", sep = "")
		
		# calculate and print speckle disk diameter base on these values
		hypotheses.combined$instrument$D.disk <- hypotheses.combined$instrument$D.speckle * 0.5 * 100 * hypotheses.combined$instrument$D / hypotheses.combined$instrument$r0
		cat("D.disk = ", hypotheses.combined$instrument$D.disk, " px\n", sep = "")
		
		# the *real* radius of the speckle disk
		# (note the factor 1/3 at the very end... this is due to
		# VLT/NACO's good AO-performance)
		hypotheses.combined$info$radius <- 5*round(1/5 * hypotheses.combined$instrument$D.disk / 2 * 1/3)
		
		# however, ask the user for the radius up to which to analyze...
		# maybe there is no need for analysis far out;
		# note, that a large radius dramatically increases computation time,
		# so the smallest possible radius should always be used!
		if (interactive() && toupper(substr(readline(paste("Suggested radius for analysis around the star is ", hypotheses.combined$info$radius, " px. Ok? [Y/n] ", sep = "")), 1, 1)) == "N")
			repeat {
				value <- as.numeric(readline("Enter radius. radius [px] = "))
				if (!is.na(value)) {
					hypotheses.combined$info$radius <- value
					break
				}
			}
		else
			cat("radius = ", hypotheses.combined$info$radius, " px\n", sep = "")
		
		# set the dimensions of the resulting hypothesis image
		# based on the given radius
		nrowcol <- 2 * hypotheses.combined$info$radius + 1
		hypotheses.combined$data <- matrix(data = 0, nrow = nrowcol, ncol = nrowcol)
			
		# read out object name
		object.name <- header.find(header, "OBJECT")
		if (length(object.name) > 0)
			hypotheses.combined$object$name <- object.name
		
		# ask the user for the distance of the imaged object [pc]
		# with this information, distances in the evaluation can be given in AU
		cat("You may additionally enter the object's parallax [mas] (press <Enter> to skip).\n")
		parallax <- as.numeric(readline("parallax [mas] = "))
		if (!is.na(parallax) && parallax > 0)
			hypotheses.combined$object$distance.pc <- 1000/parallax
	}
	
	# this is just for the calculation of the remaining time and
	# some logging purposes
	time.string <- format(time.start, "%Y-%m-%d-%H%M%S")
	time.elapsed.start <- hypotheses.combined$info$time.elapsed
	time.elapsed.session.start <- proc.time()[3]
	
	# we need this to check if we already have created
	# the necessary masks for the ringscale-step
	# (see below)
	masks.created <- FALSE
	
	# handle each FITS file one after another
	# if this session was resumed, already processed files
	# will be skipped
	while (any(!(hypotheses.combined$info$files))) {
		# get filename
		file <- names(hypotheses.combined$info$files[hypotheses.combined$info$files == FALSE][1])
		
		# count of processed files, including the current one
		file.count <- sum(hypotheses.combined$info$files == TRUE) + 1
		
		# get the regions of interest (ROIs)
		images <- rois.fromFITS(filename = file, radius = hypotheses.combined$info$radius)
		
		# this is our estimate for the total number of images
		# however, as files may contain a different amount of
		# images, this may not be correct for all data sets
		image.count.total <- image.count + sum(hypotheses.combined$info$files == FALSE) * dim(images)[3]
		
		# handle each ROI one after another
		for (i in 1:(dim(images)[3])) {
			image.count <- image.count + 1
			
			image <- images[,,i]
			
			if (file.count == 1 && image.count == 1)
				cat("Starting analysis ...\n")
			
			image.dim <- dim(hypotheses.combined$"data")
			
			# create masks for the ringscale step;
			# we only have to do this once, and
			# so we can speed ringscale up a bit
			if (!masks.created) {
				rings <- ceiling(max(image.dim[1],image.dim[2])/2)
				ring.masks <- array(data = NA, dim = c(image.dim[1],image.dim[2],rings+1))
				ring.rest <- matrix(data = TRUE, image.dim[1],image.dim[2])
				for (i in 1:rings) {
					ring.masks[,,i] <- mask.circle(width = image.dim[1], height = image.dim[2], radius.outer = i, radius.inner = i-1)
					ring.rest <- ring.rest & !ring.masks[,,i]
				}
				ring.masks[,,rings+1] <- ring.rest
				masks.created <- TRUE
			}
			
			image.hypotheses <- roi.candidates(image, ring.masks = ring.masks, D.speckle = hypotheses.combined$instrument$D.speckle)
			
			# sum up the hypotheses from the single images
			if (!is.null(image.hypotheses))
				for (j in 1:dim(image.hypotheses)[1])
					hypotheses.combined$data[image.hypotheses[j,1],image.hypotheses[j,2]] <- hypotheses.combined$data[image.hypotheses[j,1],image.hypotheses[j,2]] + 1
			
			# update the time logs and estimate the remaining computation time
			time.elapsed.session <- proc.time()[3] - time.elapsed.session.start
			hypotheses.combined$info$time.elapsed <- time.elapsed.start + time.elapsed.session
			time.left <- (image.count.total/image.count - 1) * hypotheses.combined$info$time.elapsed
			
			# show progress bar
			progress.Bar(percent = image.count/image.count.total, bar.length = options()$width-16, bar.char = "=", bar.right = progress.Meter(image.count, left = "", right = ""), bar.end = paste(" ", fill(minsec(time.left), to = 8, with = " "), sep = ""), bar.left = "|")
		}
		
		# update values for result
		hypotheses.combined$info$time.last = Sys.time()
		hypotheses.combined$info$image.count <- image.count
		hypotheses.combined$info$image.count.total <- image.count.total
		hypotheses.combined$info$files[names(hypotheses.combined$info$files) == file] <- TRUE
		
		# save current progress
		if (all(hypotheses.combined$info$files)) {
			file.remove(paste(hypotheses.combined$info$result.filename, ".ringscale.part", sep = ""))
			save(list = "hypotheses.combined", file = hypotheses.combined$info$result.filename)
		} else
			save(list = "hypotheses.combined", file = paste(hypotheses.combined$info$result.filename, ".ringscale.part", sep = ""))
	}
	
	cat("\nDone. The results have been saved in the file\n", hypotheses.combined$info$result.filename, "\n", sep = "")
	
	# call the evaluation function for the created results, if desired
	if (evaluate)
		evaluate(filename = hypotheses.combined$info$result.filename)
}