#' check if a bot has permissions to access page
#' @param permissions data.frame generated by \code{rt_get_permissions()} with
#'  three columns: 'useragent' (name of the bot), 'permission' (permission
#'  granted: Disallow/Allow) and 'value' (path for which permission applies)
#' @param bot name of the bot, defaults to "*"
#' @seealso \link{paths_allowed}
#' @param path path for which to check bot's permission, defaults to "/"
#' @return TRUE permission granted; FALSE permission denied; NA path_allowed()
#' cannot determine if path is allowed or not allowed, will result in a warning
#' as well
#' @keywords internal

path_allowed <- function(permissions, path="/", bot="*"){
  # checking and initialization
  stopifnot(length(bot)==1)
  if( is.null(bot) | bot=="" | is.na(bot) ) bot <- "*"
  perm_sanitized <- sanitize_permissions(permissions)
  path <- sanitize_path(path)

  # subsetting to permissions relevant to bot
  perm_applicable <-
    perm_sanitized[
      grepl("\\*", perm_sanitized$useragent) | tolower(bot)==tolower(perm_sanitized$useragent),
      ]

  # checking which permissions are applicable to path
  if( dim(perm_applicable)[1] > 0 ){
    perm_applicable <-
      perm_applicable[
        sapply(perm_applicable$value, grepl, x=path),
        ]
  }

  # deciding upon rules
  # no permissions --> TRUE
  if( dim(perm_applicable)[1]==0 ){
    return(TRUE)
  }
  # only disallows --> FALSE
  if ( all(grepl("disallow", perm_applicable$field, ignore.case = TRUE)) ){
    return(FALSE)
  }
  # only allows --> TRUE
  if ( all(grepl("^allow", perm_applicable$field, ignore.case = TRUE)) ){
    return(TRUE)
  }
  # diverse permissions but bot specific all disallow
  if ( all(grepl("disallow", with(perm_applicable, field[tolower(useragent)==tolower(bot)]), ignore.case = TRUE)) ){
    return(FALSE)
  }
  # diverse permissions but bot specific all allow
  if ( all(grepl("^allow", with(perm_applicable, field[tolower(useragent)==tolower(bot)]), ignore.case = TRUE)) ){
    return(TRUE)
  }
  # diverse permissions --> longest permision wins
  if (
    any(grepl("disallow", perm_applicable$field, ignore.case = TRUE)) &
    any(grepl("^allow", perm_applicable$field, ignore.case = TRUE))
  ){
    perm_path_lengths <- stringr::str_count(perm_applicable$value)
    iffer <- perm_path_lengths == max(perm_path_lengths)
    # take longest permission applicable and return TRUE if it Allows, false if it Disallows
    if( sum(iffer) == 1 ){
      return(
        !grepl("disallow", perm_applicable[iffer,]$field, ignore.case = TRUE)
      )
    }
    if( sum(iffer) > 1 ){
      if( any( grepl("disallow", perm_applicable[iffer,]$field, ignore.case = TRUE) ) ){
        return(FALSE)
      }
    }
  }
  # next missing case ...

  # message and return in case that case is not covered
  warning(
    paste0(
      " robotstxt::path_allowed() says:",
      " Encountered problems while trying to determine bot permissions, returning NA instead of TRUE/FALSE",
      sep="\n"
    )
  )
  #dev# return( named_list( path, bot, perm_applicable ) )
  return(NA)
}