% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sofar.R
\name{sofar}
\alias{sofar}
\title{Sparse orthogonal factor regression}
\usage{
sofar(Y, X, nrank = 1, ic.type = c("GIC", "AIC", "BIC", "GCV"),
      modstr = list(), control = list(), screening = TRUE)
}
\arguments{
\item{Y}{response matrix}

\item{X}{covariate matrix}

\item{nrank}{an integer specifying the desired rank/number of factors}

\item{ic.type}{select tuning method; the default is GIC}

\item{modstr}{a list of internal model parameters controlling the model
fitting}

\item{control}{a list of internal computation parameters controlling
optimization}

\item{screening}{If TRUE, marginal screening via lasso is performed before
sofar fitting.}
}
\value{
A \code{sofar} object containing
  \item{call}{original function call}
  \item{Y}{input response matrix}
  \item{X}{input predictor matrix}
  \item{Upath}{solution path of U}
  \item{Dpath}{solution path of D}
  \item{Vpath}{solution path of D}
  \item{U}{estimated left singular matrix that is orthogonal (factor weights)}
  \item{V}{estimated right singular matrix that is orthogonal (factor loadings)}
  \item{D}{estimated singular values}
  \item{rank}{estimated rank}
}
\description{
Compute solution paths of sparse orthogonal factor regression
}
\details{
The model parameters can be specified through argument \code{modstr}.
The available elements include
\itemize{
    \item{mu}: parameter in the augmented Lagrangian function.
    \item{mugamma}: increament of mu along iterations to speed up
        computation.
    \item{WA}: weight matrix for A.
    \item{WB}: weight matrix for B.
    \item{Wd}: weight matrix for d.
    \item{wgamma}: power parameter in constructing adaptive weights.
}
The model fitting can be controled through argument \code{control}.
The avilable elements include
\itemize{
   \item{nlam}: number of lambda triplets to be used.
   \item{lam.min.factor}: set the smallest lambda triplets as a fraction of the
    estimation lambda.max triplets.
   \item{lam.max.factor}: set the largest lambda triplets as a multiple of the
    estimation lambda.max triplets.
   \item{lam.AB.factor}: set the relative penalty level between A/B and D.
   \item{penA,penB,penD}: if TRUE, penalty is applied.
   \item{lamA}: sequence of tuning parameters for A.
   \item{lamB}: sequence of tuning parameters for B.
   \item{lamD}: sequence of tuning parameters for d.
   \item{methodA}: penalty for penalizing A.
   \item{methodB}: penalty for penalizing B.
   \item{epsilon}: convergence tolerance.
   \item{maxit}: maximum number of iterations.
   \item{innerEpsilon}: convergence tolerance for inner subroutines.
   \item{innerMaxit}: maximum number of iterations for inner subroutines.
   \item{sv.tol}: tolerance for singular values.
}
}
\examples{
\dontrun{
library(rrpack)
## Simulate data from a sparse factor regression model
p <- 100; q <- 50; n <- 100; nrank <- 3
mydata <- rrr.sim1(n, p, q, nrank, s2n = 1,
                   sigma = NULL, rho_X = 0.5, rho_E = 0.3)
Y <- mydata$Y
X <- mydata$X

fit1 <- sofar(Y, X, ic.type = "GIC", nrank = nrank + 2,
              control = list(methodA = "adlasso", methodB = "adlasso"))
summary(fit1)
plot(fit1)

fit1$U
crossprod(fit1$U) #check orthogonality
fit1$V
crossprod(fit1$V) #check orthogonality
}

}
\references{
Y. Uematsu, K. Chen, Y. Fan, J. Lv, and W. Lin. (2017) SOFAR:
large-scale association network learning. \emph{arXiv:1704.08349}.
}
