% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loo.R
\name{loo.stanreg}
\alias{loo.stanreg}
\alias{loo}
\alias{waic}
\alias{waic.stanreg}
\alias{kfold}
\alias{compare_models}
\alias{loo_model_weights.stanreg_list}
\alias{loo_model_weights}
\title{Information criteria and cross-validation}
\usage{
\method{loo}{stanreg}(x, ..., cores = getOption("mc.cores", 1),
  save_psis = FALSE, k_threshold = NULL)

\method{waic}{stanreg}(x, ...)

kfold(x, K = 10, save_fits = FALSE, folds = NULL)

compare_models(..., loos = list(), detail = FALSE)

\method{loo_model_weights}{stanreg_list}(x, ...,
  cores = getOption("mc.cores", 1), k_threshold = NULL)
}
\arguments{
\item{x}{For \code{loo}, \code{waic}, and \code{kfold} methods, a fitted
  model object returned by one of the rstanarm modeling functions. See
  \link{stanreg-objects}.

  For \code{loo_model_weights}, \code{x} should be a "stanreg_list"
  object, which is a list of fitted model objects created by
  \code{\link{stanreg_list}}.}

\item{...}{For \code{compare_models}, \code{...} should contain two or more
  objects returned by the \code{loo}, \code{kfold}, or \code{waic} method
  (see the \strong{Examples} section, below).

  For \code{loo_model_weights}, \code{...} should contain arguments
  (e.g. \code{method}) to pass to the default
  \code{\link[loo]{loo_model_weights}} method from the \pkg{loo} package.}

\item{cores, save_psis}{Passed to \code{\link[loo]{loo}}.}

\item{k_threshold}{Threshold for flagging estimates of the Pareto shape
parameters \eqn{k} estimated by \code{loo}. See the \emph{How to proceed
when \code{loo} gives warnings} section, below, for details.}

\item{K}{For \code{kfold}, the number of subsets (folds)
into which the data will be partitioned for performing
\eqn{K}-fold cross-validation. The model is refit \code{K} times, each time
leaving out one of the \code{K} folds. If \code{K} is equal to the total
number of observations in the data then \eqn{K}-fold cross-validation is
equivalent to exact leave-one-out cross-validation.}

\item{save_fits}{For \code{kfold}, if \code{TRUE}, a component \code{'fits'}
is added to the returned object to store the cross-validated
\link[=stanreg-objects]{stanreg} objects and the indices of the omitted
observations for each fold. Defaults to \code{FALSE}.}

\item{folds}{For \code{kfold}, an optional integer vector with one element
  per observation in the data used to fit the model. Each element of the
  vector is an integer in \code{1:K} indicating to which of the \code{K}
  folds the corresponding observation belongs. There are some convenience
  functions available in the \pkg{loo} package that create integer vectors to
  use for this purpose (see the \strong{Examples} section below and also the
  \link[loo]{kfold-helpers} page).

  If \code{folds} is not specified then the default is to call
  \code{loo::\link[loo]{kfold_split_random}} to randomly partition the data
  into \code{K} subsets of equal (or as close to equal as possible) size.}

\item{loos}{For \code{compare_models}, a list of two or more objects returned
by the \code{loo}, \code{kfold}, or \code{waic} method. This argument can
be used as an alternative to passing these objects via \code{...}.}

\item{detail}{For \code{compare_models}, if \code{TRUE} then extra
information about each model (currently just the model formulas) will be
printed with the output.}
}
\value{
The structure of the objects returned by \code{loo} and \code{waic}
  methods are documented in detail in the \strong{Value} section in
  \code{\link[loo]{loo}} and \code{\link[loo]{waic}} (from the \pkg{loo}
  package).

\code{kfold} returns an object with classes 'kfold' and 'loo' that
  has a similar structure as the objects returned by the \code{loo} and
  \code{waic} methods.

\code{compare_models} returns a vector or matrix with class
  'compare.loo'. See the \strong{Comparing models} section below for more
  details.
}
\description{
For models fit using MCMC, compute approximate leave-one-out
  cross-validation (LOO, LOOIC) or, less preferably, the Widely Applicable
  Information Criterion (WAIC) using the \pkg{\link[=loo-package]{loo}}
  package. Functions for \eqn{K}-fold cross-validation, model comparison,
  and model weighting/averaging are also provided. \strong{Note}:
  these functions are not guaranteed to work properly unless the \code{data}
  argument was specified when the model was fit. Also, as of \pkg{loo}
  version \code{2.0.0} the default number of cores is now only 1,  but we
  recommend using as many (or close to as many) cores as possible by setting
  the \code{cores} argument or using \code{options(mc.cores = VALUE)} to set
  it for an entire session.
}
\section{Approximate LOO CV}{
 The \code{loo} method for stanreg objects
  provides an interface to the \pkg{\link[=loo-package]{loo}} package for
  approximate leave-one-out cross-validation (LOO). The LOO Information
  Criterion (LOOIC) has the same purpose as the Akaike Information Criterion
  (AIC) that is used by frequentists. Both are intended to estimate the
  expected log predictive density (ELPD) for a new dataset. However, the AIC
  ignores priors and assumes that the posterior distribution is multivariate
  normal, whereas the functions from the \pkg{loo} package do not make this
  distributional assumption and integrate over uncertainty in the parameters.
  This only assumes that any one observation can be omitted without having a
  major effect on the posterior distribution, which can be judged using the
  diagnostic plot provided by the \code{\link[loo]{plot.loo}} method and the
  warnings provided by the \code{\link[loo]{print.loo}} method (see the
  \emph{How to Use the rstanarm Package} vignette for an example of this
  process).

  \subsection{How to proceed when \code{loo} gives warnings (k_threshold)}{
  The \code{k_threshold} argument to the \code{loo} method for \pkg{rstanarm}
  models is provided as a possible remedy when the diagnostics reveal
  problems stemming from the posterior's sensitivity to particular
  observations. Warnings about Pareto \eqn{k} estimates indicate observations
  for which the approximation to LOO is problematic (this is described in
  detail in Vehtari, Gelman, and Gabry (2017) and the
  \pkg{\link[=loo-package]{loo}} package documentation). The
  \code{k_threshold} argument can be used to set the \eqn{k} value above
  which an observation is flagged. If \code{k_threshold} is not \code{NULL}
  and there are \eqn{J} observations with \eqn{k} estimates above
  \code{k_threshold} then when \code{loo} is called it will refit the
  original model \eqn{J} times, each time leaving out one of the \eqn{J}
  problematic observations. The pointwise contributions of these observations
  to the total ELPD are then computed directly and substituted for the
  previous estimates from these \eqn{J} observations that are stored in the
  object created by \code{loo}.

  \strong{Note}: in the warning messages issued by \code{loo} about large
  Pareto \eqn{k} estimates we recommend setting \code{k_threshold} to at
  least \eqn{0.7}. There is a theoretical reason, explained in Vehtari,
  Gelman, and Gabry (2017), for setting the threshold to the stricter value
  of \eqn{0.5}, but in practice they find that errors in the LOO
  approximation start to increase non-negligibly when \eqn{k > 0.7}.
  }
}

\section{K-fold CV}{
 The \code{kfold} function performs exact \eqn{K}-fold
  cross-validation. First the data are randomly partitioned into \eqn{K}
  subsets of equal (or as close to equal as possible) size (unless the folds
  are specified manually). Then the model is refit \eqn{K} times, each time
  leaving out one of the \eqn{K} subsets. If \eqn{K} is equal to the total
  number of observations in the data then \eqn{K}-fold cross-validation is
  equivalent to exact leave-one-out cross-validation (to which \code{loo} is
  an efficient approximation). The \code{compare_models} function is also
  compatible with the objects returned by \code{kfold}.
}

\section{Comparing models}{
 \code{compare_models} is a method for the
  \code{\link[loo]{compare}} function in the \pkg{loo} package that
  performs some extra checks to make sure the \pkg{rstanarm} models are
  suitable for comparison. These extra checks include verifying that all
  models to be compared were fit using the same outcome variable and
  likelihood family.

  If exactly two models are being compared then \code{compare_models} returns
  a vector containing the difference in expected log predictive density
  (ELPD) between the models and the standard error of that difference (the
  documentation for \code{\link[loo]{compare}} in the \pkg{loo}
  package has additional details about the calculation of the standard error
  of the difference). The difference in ELPD will be negative if the expected
  out-of-sample predictive accuracy of the first model is higher. If the
  difference is be positive then the second model is preferred.

  If more than two models are being compared then \code{compare_models}
  returns a matrix with one row per model. This matrix summarizes the objects
  and arranges them in descending order according to expected out-of-sample
  predictive accuracy. That is, the first row of the matrix will be
  for the model with the largest ELPD (smallest LOOIC).
  The columns containing the ELPD difference and the standard error of the 
  difference contain values relative to the model with the best ELPD.
  See the \strong{Details} section at the \code{\link[loo]{compare}}
  page in the \pkg{loo} package for more information.
}

\section{Model weights}{
 The \code{loo_model_weights} method can be used to
  compute model weights for a \code{"stanreg_list"} object, which is a list
  of fitted model objects made with \code{\link{stanreg_list}}. The end of
  the \strong{Examples} section has a demonstration. For details see the
  \code{\link[loo]{loo_model_weights}} documentation in the \pkg{loo}
  package.
}

\examples{
\donttest{
fit1 <- stan_glm(mpg ~ wt, data = mtcars)
fit2 <- stan_glm(mpg ~ wt + cyl, data = mtcars)

# compare on LOOIC
# (for bigger models use as many cores as possible)
loo1 <- loo(fit1, cores = 2)
print(loo1)
loo2 <- loo(fit2, cores = 2)
print(loo2)

# when comparing exactly two models, the reported 'elpd_diff'
# will be positive if the expected predictive accuracy for the
# second model is higher. the approximate standard error of the
# difference is also reported.
compare_models(loo1, loo2)
compare_models(loos = list(loo1, loo2)) # can also provide list

# when comparing three or more models they are ordered by
# expected predictive accuracy. elpd_diff and se_diff are relative
# to the model with best elpd_loo (first row)
fit3 <- stan_glm(mpg ~ disp * as.factor(cyl), data = mtcars)
loo3 <- loo(fit3, cores = 2, k_threshold = 0.7)
compare_models(loo1, loo2, loo3)

# setting detail=TRUE will also print model formulas
compare_models(loo1, loo2, loo3, detail=TRUE)

# Computing model weights
model_list <- stanreg_list(fit1, fit2, fit3)
loo_model_weights(model_list, cores = 2) # can specify k_threshold=0.7 if necessary

# if you have already computed loo then it's more efficient to pass a list
# of precomputed loo objects than a "stanreg_list", avoiding the need
# for loo_models weights to call loo() internally
loo_list <- list(fit1 = loo1, fit2 = loo2, fit3 = loo3) # names optional (affects printing)
loo_model_weights(loo_list)

# 10-fold cross-validation
(kfold1 <- kfold(fit1, K = 10))
kfold2 <- kfold(fit2, K = 10)
compare_models(kfold1, kfold2, detail=TRUE)

# Cross-validation stratifying by a grouping variable
# (note: might get some divergences warnings with this model but 
# this is just intended as a quick example of how to code this)
library(loo)
fit4 <- stan_lmer(mpg ~ disp + (1|cyl), data = mtcars)
table(mtcars$cyl)
folds_cyl <- kfold_split_stratified(K = 3, x = mtcars$cyl)
table(cyl = mtcars$cyl, fold = folds_cyl)
kfold4 <- kfold(fit4, K = 3, folds = folds_cyl)
}

}
\references{
Vehtari, A., Gelman, A., and Gabry, J. (2017). Practical
  Bayesian model evaluation using leave-one-out cross-validation and WAIC.
  \emph{Statistics and Computing}. 27(5), 1413--1432.
  doi:10.1007/s11222-016-9696-4. arXiv preprint:
  \url{http://arxiv.org/abs/1507.04544/}

  Yao, Y., Vehtari, A., Simpson, D., and Gelman, A. (2018) Using
  stacking to average Bayesian predictive distributions. \emph{Bayesian
  Analysis}, advance publication,  doi:10.1214/17-BA1091.
  (\href{https://projecteuclid.org/euclid.ba/1516093227}{online}).

Gabry, J., Simpson, D., Vehtari, A., Betancourt, M., and Gelman,
  A. (2018). Visualization in Bayesian workflow. \emph{Journal of the Royal
  Statistical Society Series A}, accepted for publication. arXiv preprint:
  \url{http://arxiv.org/abs/1709.01449}.
}
\seealso{
\itemize{
  \item The new \href{http://mc-stan.org/loo/articles/}{\pkg{loo} package vignettes}
  and various \href{http://mc-stan.org/rstanarm/articles/}{\pkg{rstanarm} vignettes}
  for more examples using \code{loo} and related functions with \pkg{rstanarm} models.
  \item \code{\link[loo]{pareto-k-diagnostic}} in the \pkg{loo} package for
  more on Pareto \eqn{k} diagnostics.
  \item \code{\link{log_lik.stanreg}} to directly access the pointwise
  log-likelihood matrix.
}
}
