% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make_split_fun.R
\name{make_split_fun}
\alias{make_split_fun}
\title{Create a Custom Splitting Function}
\usage{
make_split_fun(pre = list(), core_split = NULL, post = list())
}
\arguments{
\item{pre}{list.  Zero or  more functions  which  operate on  the
incoming data and return a new data frame that should split via
\code{core_split}. They will be called on the data in the order they
appear in the list.}

\item{core_split}{function or NULL. If not NULL,  a function which
accepts the same arguments  do_base_split does, and returns the
same type of named list.   Custom functions which override this
behavior cannot be used in column splits.}

\item{post}{list. Zero or  more functions which should be called on
the list output by splitting.}
}
\value{
A function for use as a custom split function.
}
\description{
Create a Custom Splitting Function
}
\details{
Custom split  functions can be  thought of  as (up to)  3 different
types of manipulations of the splitting process
\enumerate{
\item Preprocessing of the incoming data to be split
\item (Row-splitting only) Customization of the core mapping of incoming data to facets, and
\item Postprocessing operations on the set of facets (groups) generated by the split.
}

This  function  provides  an   interface  to  create  custom  split
functions by implementing and specifying sets of operations in each
of those classes of customization independently.

Preprocessing  functions (1),  must  accept:  \code{df}, \code{spl},  \code{vals},
\code{labels},  and  can  optionally accept  \code{.spl_context}.  They  then
manipulate  \code{df} (the  incoming data  for the  split) and  return a
modified data.frame.  This modified  data.frame \emph{must}  contain all
columns present in the incoming  data.frame, but can add columns if
necessary (though we note that these  new columns cannot be used in
the layout as split or analysis variables, because they will not be
present when validity checking is done).

The  preprocessing   component  is   useful  for  things   such  as
manipulating factor  levels, e.g.,  to trim  unobserved ones  or to
reorder levels based on observed counts, etc.

Customization of core splitting (2)  is currently only supported in
row  splits.  Core  splitting functions  override  the  fundamental
splitting procedure,  and are only  necessary in rare  cases. These
must  accept  \code{spl}, \code{df},  \code{vals},  \code{labels},  and can  optionally
accept \code{.spl_context}. They must return a named list with elements,
all of  the same  length, as follows:  - \code{datasplit}  (containing a
list of data.frames), -  \code{values} containing values associated with
the   facets,   which   must   be   \code{character}   or   \code{SplitValue}
objects. These  values will  appear in the  paths of  the resulting
table.  - \code{labels} containing  the character labels associated with
\code{values}

Postprocessing functions  (3) must  accept the  result of  the core
split  as  their  first  argument  (which  as  of  writing  can  be
anything), in addition  to \code{spl}, and \code{fulldf},  and can optionally
accept \code{.spl_context}. They must each  return a modified version of
the same structure specified above for core splitting.

In both the pre- and  post-processing cases, multiple functions can
be specified. When this happens,  they are applied sequentially, in
the order they  appear in the list passed to  the relevant argument
(\code{pre} and \code{post}, respectively).
}
\examples{

mysplitfun <- make_split_fun(pre = list(drop_facet_levels),
    post = list(add_overall_facet("ALL", "All Arms")))


basic_table(show_colcounts = TRUE) \%>\%
   split_cols_by("ARM", split_fun = mysplitfun) \%>\%
   analyze("AGE") \%>\%
   build_table(subset(DM, ARM \%in\% c("B: Placebo", "C: Combination")))

## post (and pre) arguments can take multiple functions, here
## we add an overall facet and the reorder the facets
reorder_facets <- function(splret, spl, fulldf, ...) {
  ord <- order(names(splret$values))
  make_split_result(splret$values[ord],
                      splret$datasplit[ord],
                      splret$labels[ord])
}

mysplitfun2 <-  make_split_fun(pre = list(drop_facet_levels),
                             post = list(add_overall_facet("ALL", "All Arms"),
                                         reorder_facets))
basic_table(show_colcounts = TRUE) \%>\%
   split_cols_by("ARM", split_fun = mysplitfun2) \%>\%
   analyze("AGE") \%>\%
   build_table(subset(DM, ARM \%in\% c("B: Placebo", "C: Combination")))

very_stupid_core <- function(spl, df, vals, labels, .spl_context) {
    make_split_result(c("stupid", "silly"),
                      datasplit = list(df[1:10,], df[11:30,]),
                      labels = c("first 10", "second 20"))
}

dumb_30_facet <- add_combo_facet("dumb",
                                 label = "thirty patients",
                                 levels = c("stupid", "silly"))
nonsense_splfun <-  make_split_fun(core_split = very_stupid_core,
                                   post = list(dumb_30_facet))

## recall core split overriding is not supported in column space
## currently, but we can see it in action in row space

lyt_silly <- basic_table() \%>\%
      split_rows_by("ARM", split_fun = nonsense_splfun) \%>\%
      summarize_row_groups() \%>\%
      analyze("AGE")
silly_table <- build_table(lyt_silly, DM)
silly_table
}
\seealso{
Other make_custom_split: 
\code{\link{add_combo_facet}()},
\code{\link{drop_facet_levels}()},
\code{\link{make_split_result}()},
\code{\link{trim_levels_in_facets}()}
}
\concept{make_custom_split}
