% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ru_sampling_rcpp.R
\name{ru_rcpp}
\alias{ru_rcpp}
\title{Generalized ratio-of-uniforms sampling using C++ via Rcpp}
\usage{
ru_rcpp(logf, ..., n = 1, d = 1, init = NULL, trans = c("none", "BC",
  "user"), phi_to_theta = NULL, log_j = NULL, user_args = list(),
  lambda = rep(1L, d), lambda_tol = 1e-06, gm = NULL, rotate = ifelse(d
  == 1, FALSE, TRUE), lower = rep(-Inf, d), upper = rep(Inf, d), r = 1/2,
  ep = 0L, a_algor = if (d == 1) "nlminb" else "optim",
  b_algor = c("nlminb", "optim"), a_method = c("Nelder-Mead", "BFGS", "CG",
  "L-BFGS-B", "SANN", "Brent"), b_method = c("Nelder-Mead", "BFGS", "CG",
  "L-BFGS-B", "SANN", "Brent"), a_control = list(), b_control = list(),
  var_names = NULL)
}
\arguments{
\item{logf}{An external pointer to a compiled C++ function returning the
log of the target density \eqn{f}.
This function should return \code{-Inf} when the density is zero.
See the \code{vignette("rust-using-rcpp-vignette", package = "rust")},
particularly the Section
\strong{Providing a C++ function to \code{ru_rcpp}}, for details.}

\item{...}{Further arguments to be passed to \code{logf} and related
functions.}

\item{n}{A numeric scalar.  Number of simulated values required.}

\item{d}{A numeric scalar. Dimension of f.}

\item{init}{A numeric vector. Initial estimates of the mode of \code{logf}.
If \code{trans = "BC"} or \code{trans = "user"} this is \emph{after}
Box-Cox transformation or user-defined transformation, but \emph{before}
any rotation of axes.}

\item{trans}{A character scalar. "none" for no transformation, "BC" for
Box-Cox transformation, "user" for a user-defined transformation.
If \code{trans = "user"} then the transformation should be specified
using \code{phi_to_theta} and \code{log_j} and \code{user_args} may be
used to pass arguments to \code{phi_to_theta} and \code{log_j}.}

\item{phi_to_theta}{An external pointer to a compiled C++ function returning
(the inverse) of the transformation from theta to phi used to ensure
positivity of phi prior to Box-Cox transformation.  The argument is
phi and the returned value is theta.  If \code{phi_to_theta}
is undefined at the input value then the function should return NA.}

\item{log_j}{An external pointer to a compiled C++ function returning the
log of the Jacobian of the transformation from theta to phi, i.e. based on
derivatives of phi with respect to theta. Takes theta as its argument.}

\item{user_args}{A list of numeric components. If \code{trans = ``user''}
then \code{user_args} is a list providing arguments to the user-supplied
functions \code{phi_to_theta} and \code{log_j}.}

\item{lambda}{Either
\itemize{
  \item {A numeric vector.  Box-Cox transformation parameters, or}
  \item {A list with components}
  \describe{
    \item{lambda}{A numeric vector.  Box-Cox parameters (required).}
    \item{gm}{A numeric vector.  Box-cox scaling parameters (optional).
      If supplied this overrides any \code{gm} supplied by the individual
      \code{gm} argument described below.}
    \item{init_psi}{A numeric vector.  Initial estimate of mode \emph{after}
      Box-Cox transformation (optional).}
    \item{sd_psi}{A numeric vector.  Estimates of the marginal standard
      deviations of the Box-Cox transformed variables (optional).}
    \item{phi_to_theta}{as above (optional).}
    \item{log_j}{as above (optional).}
    \item{user_args}{as above (optional).}
  }
  This list may be created using \code{\link{find_lambda_one_d_rcpp}}
  (for \code{d} = 1) or \code{\link{find_lambda_rcpp}} (for any \code{d}).
}}

\item{lambda_tol}{A numeric scalar.  Any values in lambda that are less
than lambda_tol in magnitude are set to zero.}

\item{gm}{A numeric vector. Box-cox scaling parameters (optional). If
\code{lambda$gm} is supplied in input list \code{lambda} then
\code{lambda$gm} is used, not \code{gm}.}

\item{rotate}{A logical scalar. If TRUE (\code{d} > 1 only) use Choleski
rotation.  If d = 1 and \code{rotate} = TRUE then rotate will be set to
FALSE with a warning.}

\item{lower, upper}{Numeric vectors.  Lower/upper bounds on the arguments of
the function \emph{after} any transformation from theta to phi implied by
the inverse of \code{phi_to_theta}. If \code{rotate = FALSE} these
are used in all of the optimizations used to construct the bounding box.
If \code{rotate = TRUE} then they are use only in the first optimisation
to maximise the target density.`
If \code{trans = "BC"} components of \code{lower} that are negative are
set to zero without warning and the bounds implied after the Box-Cox
transformation are calculated inside \code{ru}.}

\item{r}{A numeric scalar.  Parameter of generalized ratio-of-uniforms.}

\item{ep}{A numeric scalar.  Controls initial estimates for optimizations
to find the b-bounding box parameters.  The default (\code{ep}=0)
corresponds to starting at the mode of \code{logf} small positive values
of \code{ep} move the constrained variable slightly away from the mode in
the correct direction.  If \code{ep} is negative its absolute value is
used, with no warning given.}

\item{a_algor, b_algor}{Character scalars.  Either "nlminb" or "optim".
Respective optimization algorithms used to find a(r) and (bi-(r), bi+(r)).}

\item{a_method, b_method}{Character scalars.  Respective methods used by
\code{optim} to find a(r) and (bi-(r), bi+(r)).  Only used if \code{optim}
is the chosen algorithm.  If \code{d} = 1 then \code{a_method} and
\code{b_method} are set to \code{"Brent"} without warning.}

\item{a_control, b_control}{Lists of control arguments to \code{optim} or
\code{nlminb} to find a(r) and (bi-(r), bi+(r)) respectively.}

\item{var_names}{A character vector.  Names to give to the column(s) of
the simulated values.}
}
\value{
An object of class "ru" is a list containing the following
  components:
    \item{sim_vals}{An \code{n} by \code{d} matrix of simulated values.}
    \item{box}{A (2 * \code{d} + 1) by \code{d} + 2 matrix of
      ratio-of-uniforms bounding box information, with row names indicating
      the box parameter.  The columns contain
      \describe{
        \item{column 1}{values of box parameters.}
        \item{columns 2 to (2+\code{d}-1)}{values of variables at which
         these box parameters are obtained.}
        \item{column 2+\code{d}}{convergence indicators.}
      }
      Scaling of f within \code{ru} and relocation of the
      mode to the origin means that the first row of \code{box} will always
      be \code{c(1, rep(0, d))}.
    }
    \item{pa}{A numeric scalar.  An estimate of the probability of
      acceptance.}
    \item{d}{A numeric scalar.  The dimension of \code{logf}.}
    \item{logf}{A function. \code{logf} supplied by the user, but
      with f scaled by the maximum of the target density used in the
      ratio-of-uniforms method (i.e. \code{logf_rho}), to avoid numerical
      problems in contouring f in \code{\link{plot.ru}} when
      \code{d = 2}.}
    \item{logf_rho}{A function. The target function actually used in the
      ratio-of-uniforms algorithm.}
    \item{sim_vals_rho}{An \code{n} by \code{d} matrix of values simulated
      from the function used in the ratio-of-uniforms algorithm.}
    \item{logf_args}{A list of further arguments to \code{logf}.}
    \item{logf_rho_args}{A list of further arguments to \code{logf_rho}.
      Note: this component is returned by \code{ru_rcpp} but not
      by \code{ru}.}
    \item{f_mode}{The estimated mode of the target density f, after any
      Box-Cox transformation and/or user supplied transformation, but before
      mode relocation.}
}
\description{
Uses the generalized ratio-of-uniforms method to simulate from a
distribution with log-density \eqn{log f} (up to an additive constant).
\eqn{f} must be bounded, perhaps after a transformation of variable.
The file file `user_fns.cpp` that is sourced before running the examples
below is available at the rust Github page at
\url{https://github.com/paulnorthrop/rust/blob/master/src/user_fns.cpp}.
}
\details{
If \code{trans = "none"} and \code{rotate = FALSE} then \code{ru}
  implements the (multivariate) generalized ratio of uniforms method
  described in Wakefield, Gelfand and Smith (1991) using a target
  density whose mode is relocated to the origin (`mode relocation') in the
  hope of increasing efficiency.

  If \code{trans = "BC"} then marginal Box-Cox transformations of each of
  the \code{d} variables is performed, with parameters supplied in
  \code{lambda}.  The function \code{phi_to_theta} may be used, if
  necessary, to ensure positivity of the variables prior to Box-Cox
  transformation.

  If \code{trans = "user"} then the function \code{phi_to_theta} enables
  the user to specify their own transformation.

  In all cases the mode of the target function is relocated to the origin
  \emph{after} any user-supplied transformation and/or Box-Cox
  transformation.

  If \code{d} is greater than one and \code{rotate = TRUE} then a rotation
  of the variable axes is performed \emph{after} mode relocation.  The
  rotation is based on the Choleski decomposition (see \link{chol}) of the
  estimated Hessian (computed using \code{\link[stats]{optimHess}}
  of the negated
  log-density after any user-supplied transformation or Box-Cox
  transformation.  If any of the eigenvalues of the estimated Hessian are
  non-positive (which may indicate that the estimated mode of \code{logf}
  is close to a variable boundary) then \code{rotate} is set to \code{FALSE}
  with a warning.  A warning is also given if this happens when
  \code{d} = 1.

  The default value of the tuning parameter \code{r} is 1/2, which is
  likely to be close to optimal in many cases, particularly if
  \code{trans = "BC"}.

See \code{vignette("rust-using-rcpp-vignette", package = "rust")} and
\code{vignette("rust-vignette", package = "rust")} for full details.
}
\examples{
n <- 1000

# Normal density ===================

# One-dimensional standard normal ----------------
ptr_N01 <- create_xptr("logdN01")
x <- ru_rcpp(logf = ptr_N01, d = 1, n = n, init = 0.1)

# Two-dimensional standard normal ----------------
ptr_bvn <- create_xptr("logdnorm2")
rho <- 0
x <- ru_rcpp(logf = ptr_bvn, rho = rho, d = 2, n = n,
  init = c(0, 0))

# Two-dimensional normal with positive association ===================
rho <- 0.9
# No rotation.
x <- ru_rcpp(logf = ptr_bvn, rho = rho, d = 2, n = n, init = c(0, 0),
             rotate = FALSE)

# With rotation.
x <- ru_rcpp(logf = ptr_bvn, rho = rho, d = 2, n = n, init = c(0, 0))

# Using general multivariate normal function.
ptr_mvn <- create_xptr("logdmvnorm")
covmat <- matrix(rho, 2, 2) + diag(1 - rho, 2)
x <- ru_rcpp(logf = ptr_mvn, sigma = covmat, d = 2, n = n, init = c(0, 0))

# Three-dimensional normal with positive association ----------------
covmat <- matrix(rho, 3, 3) + diag(1 - rho, 3)

# No rotation.
x <- ru_rcpp(logf = ptr_mvn, sigma = covmat, d = 3, n = n,
             init = c(0, 0, 0), rotate = FALSE)

# With rotation.
x <- ru_rcpp(logf = ptr_mvn, sigma = covmat, d = 3, n = n,
             init = c(0, 0, 0))

# Log-normal density ===================

ptr_lnorm <- create_xptr("logdlnorm")
mu <- 0
sigma <- 1
# Sampling on original scale ----------------
x <- ru_rcpp(logf = ptr_lnorm, mu = mu, sigma = sigma, d = 1, n = n,
             lower = 0, init = exp(mu))

# Box-Cox transform with lambda = 0 ----------------
lambda <- 0
x <- ru_rcpp(logf = ptr_lnorm, mu = mu, sigma = sigma, d = 1, n = n,
             lower = 0, init = exp(mu), trans = "BC", lambda = lambda)

# Equivalently, we could use trans = "user" and supply the (inverse) Box-Cox
# transformation and the log-Jacobian by hand
ptr_phi_to_theta_lnorm <- create_phi_to_theta_xptr("exponential")
ptr_log_j_lnorm <- create_log_j_xptr("neglog")
x <- ru_rcpp(logf = ptr_lnorm, mu = mu, sigma = sigma, d = 1, n = n,
  init = 0.1, trans = "user", phi_to_theta = ptr_phi_to_theta_lnorm,
  log_j = ptr_log_j_lnorm)

# Gamma (alpha, 1) density ===================

# Note: the gamma density in unbounded when its shape parameter is < 1.
# Therefore, we can only use trans="none" if the shape parameter is >= 1.

# Sampling on original scale ----------------

ptr_gam <- create_xptr("logdgamma")
alpha <- 10
x <- ru_rcpp(logf = ptr_gam, alpha = alpha, d = 1, n = n,
  lower = 0, init = alpha)

alpha <- 1
x <- ru_rcpp(logf = ptr_gam, alpha = alpha, d = 1, n = n,
  lower = 0, init = alpha)

# Box-Cox transform with lambda = 1/3 works well for shape >= 1. -----------

alpha <- 1
x <- ru_rcpp(logf = ptr_gam, alpha = alpha, d = 1, n = n,
  trans = "BC", lambda = 1/3, init = alpha)
summary(x)

# Equivalently, we could use trans = "user" and supply the (inverse) Box-Cox
# transformation and the log-Jacobian by hand

lambda <- 1/3
ptr_phi_to_theta_bc <- create_phi_to_theta_xptr("bc")
ptr_log_j_bc <- create_log_j_xptr("bc")
x <- ru_rcpp(logf = ptr_gam, alpha = alpha, d = 1, n = n,
  trans = "user", phi_to_theta = ptr_phi_to_theta_bc, log_j = ptr_log_j_bc,
  user_args = list(lambda = lambda), init = alpha)
summary(x)

\dontrun{
# Generalized Pareto posterior distribution ===================

# Sample data from a GP(sigma, xi) distribution
gpd_data <- rgpd(m = 100, xi = -0.5, sigma = 1)
# Calculate summary statistics for use in the log-likelihood
ss <- gpd_sum_stats(gpd_data)
# Calculate an initial estimate
init <- c(mean(gpd_data), 0)

n <- 1000
# Mode relocation only ----------------
ptr_gp <- create_xptr("loggp")
for_ru_rcpp <- c(list(logf = ptr_gp, init = init, d = 2, n = n,
                 lower = c(0, -Inf)), ss, rotate = FALSE)
x1 <- do.call(ru_rcpp, for_ru_rcpp)
plot(x1, xlab = "sigma", ylab = "xi")
# Parameter constraint line xi > -sigma/max(data)
# [This may not appear if the sample is far from the constraint.]
abline(a = 0, b = -1 / ss$xm)
summary(x1)

# Rotation of axes plus mode relocation ----------------
for_ru_rcpp <- c(list(logf = ptr_gp, init = init, d = 2, n = n,
                 lower = c(0, -Inf)), ss)
x2 <- do.call(ru_rcpp, for_ru_rcpp)
plot(x2, xlab = "sigma", ylab = "xi")
abline(a = 0, b = -1 / ss$xm)
summary(x2)

# Cauchy ========================

ptr_c <- create_xptr("logcauchy")

# The bounding box cannot be constructed if r < 1.  For r = 1 the
# bounding box parameters b1-(r) and b1+(r) are attained in the limits
# as x decreases/increases to infinity respectively.  This is fine in
# theory but using r > 1 avoids this problem and the largest probability
# of acceptance is obtained for r approximately equal to 1.26.

res <- ru_rcpp(logf = ptr_c, log = TRUE, init = 0, r = 1.26, n = 1000)

# Half-Cauchy ===================

ptr_hc <- create_xptr("loghalfcauchy")

# Like the Cauchy case the bounding box cannot be constructed if r < 1.
# We could use r > 1 but the mode is on the edge of the support of the
# density so as an alternative we use a log transformation.

x <- ru_rcpp(logf = ptr_hc, init = 0, trans = "BC", lambda = 0, n = 1000)
x$pa
plot(x, ru_scale = TRUE)

# Example 4 from Wakefield et al. (1991) ===================
# Bivariate normal x bivariate student-t

ptr_normt <- create_xptr("lognormt")
rho <- 0.9
covmat <- matrix(c(1, rho, rho, 1), 2, 2)
y <- c(0, 0)

# Case in the top right corner of Table 3
x <- ru_rcpp(logf = ptr_normt, mean = y, sigma1 = covmat, sigma2 = covmat,
  d = 2, n = 10000, init = y, rotate = FALSE)
x$pa

# Rotation increases the probability of acceptance
x <- ru_rcpp(logf = ptr_normt, mean = y, sigma1 = covmat, sigma2 = covmat,
  d = 2, n = 10000, init = y, rotate = TRUE)
x$pa
}

}
\references{
Wakefield, J. C., Gelfand, A. E. and Smith, A. F. M. (1991)
 Efficient generation of random variates via the ratio-of-uniforms method.
 \emph{Statistics and Computing} (1991), \strong{1}, 129-133.
 \url{http://dx.doi.org/10.1007/BF01889987}.

Eddelbuettel, D. and Francois, R. (2011). Rcpp: Seamless
 R and C++ Integration. \emph{Journal of Statistical Software},
 \strong{40}(8), 1-18.
 \url{http://www.jstatsoft.org/v40/i08/}.

Eddelbuettel, D. (2013). \emph{Seamless R and C++ Integration
 with Rcpp}, Springer, New York. ISBN 978-1-4614-6867-7.
}
\seealso{
\code{\link{ru}} for a version of \code{\link{ru_rcpp}} that
  accepts R functions as arguments.

\code{\link{summary.ru}} for summaries of the simulated values
  and properties of the ratio-of-uniforms algorithm.

\code{\link{plot.ru}} for a diagnostic plot (for \code{d} = 1
  and \code{d} = 2 only).

\code{\link{find_lambda_one_d_rcpp}} to produce (somewhat)
  automatically a list for the argument \code{lambda} of \code{ru} for the
  \code{d} = 1 case.

\code{\link{find_lambda_rcpp}} to produce (somewhat) automatically
  a list for the argument \code{lambda} of \code{ru} for any value of
  \code{d}.

\code{\link[stats]{optim}} for choices of the arguments
  \code{a_method}, \code{b_method}, \code{a_control} and \code{b_control}.

\code{\link[stats]{nlminb}} for choices of the arguments
  \code{a_control} and \code{b_control}.

\code{\link[stats]{optimHess}} for Hessian estimation.

\code{\link[base]{chol}} for the Choleski decomposition.
}
