% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ROCSS.R
\name{ROCSS}
\alias{ROCSS}
\title{Compute the Relative Operating Characteristic Skill Score}
\usage{
ROCSS(
  exp,
  obs,
  ref = NULL,
  time_dim = "sdate",
  memb_dim = "member",
  cat_dim = NULL,
  dat_dim = NULL,
  prob_thresholds = c(1/3, 2/3),
  indices_for_clim = NULL,
  cross.val = FALSE,
  ncores = NULL
)
}
\arguments{
\item{exp}{A named numerical array of either the forecast with at least time
and member dimensions, or the probabilities with at least time and category
dimensions. The probabilities can be generated by \code{s2dv::GetProbs}.}

\item{obs}{A named numerical array of either the observations with at least 
time dimension, or the probabilities with at least time and category 
dimensions. The probabilities can be generated by \code{s2dv::GetProbs}. The
dimensions must be the same as 'exp' except 'memb_dim' and 'dat_dim'.}

\item{ref}{A named numerical array of the reference forecast with at least
time and member dimensions, or the probabilities with at least time and 
category dimensions. The probability can be generated by 
\code{s2dv::GetProbs}. The dimensions must be the same as 'exp' except 
'memb_dim' and 'dat_dim'. If there is only one reference dataset, it should
not have dataset dimension. If there is corresponding reference for each 
experiment, the dataset dimension must have the same length as in 'exp'.
If 'ref' is NULL, the random forecast is used as reference forecast. The 
default value is NULL.}

\item{time_dim}{A character string indicating the name of the time dimension.
The default value is 'sdate'.}

\item{memb_dim}{A character string indicating the name of the member dimension
to compute the probabilities of the forecast and the reference forecast. The
default value is 'member'. If the data are probabilities, set memb_dim as 
NULL.}

\item{cat_dim}{A character string indicating the name of the category 
dimension that is needed when exp, obs, and ref are probabilities. The 
default value is NULL, which means that the data are not probabilities.}

\item{dat_dim}{A character string indicating the name of dataset dimension. 
The length of this dimension can be different between 'exp' and 'obs'. 
The default value is NULL.}

\item{prob_thresholds}{A numeric vector of the relative thresholds (from 0 to
1) between the categories. The default value is c(1/3, 2/3), which 
corresponds to tercile equiprobable categories.}

\item{indices_for_clim}{A vector of the indices to be taken along 'time_dim' 
for computing the thresholds between the probabilistic categories. If NULL,
the whole period is used. The default value is NULL.}

\item{cross.val}{A logical indicating whether to compute the thresholds 
between probabilistic categories in cross-validation. The default value is 
FALSE.}

\item{ncores}{An integer indicating the number of cores to use for parallel 
computation. The default value is NULL.}
}
\value{
A numerical array of ROCSS with dimensions c(nexp, nobs, cat, the rest
dimensions of 'exp' except 'time_dim' and 'memb_dim' dimensions). nexp is the
number of experiment (i.e., dat_dim in exp), and nobs is the number of 
observation (i.e., dat_dim in obs). If dat_dim is NULL, nexp and nobs are 
omitted. dimension 'cat' refers to the probabilistic category, i.e., 
\code{1 + length(prob_thresholds)}.
}
\description{
The Relative Operating Characteristic Skill Score (ROCSS; Kharin and Zwiers, 
2003) is based on the ROC curve, which gives information about the hit rates 
against the false-alarm rates for a particular category or event. The ROC
curve can be summarized with the area under the ROC curve, known as the ROC
score, to provide a skill value for each category. The ROCSS ranges between 
minus infinite and 1. A positive ROCSS value indicates that the forecast has
higher skill than the reference forecast, meaning the contrary otherwise.\cr
The function accepts either the data or the probabilities of each data as 
inputs. If there is more than one dataset, RPSS will be computed for each pair
of exp and obs data.
}
\examples{
# Use data as input
exp <- array(rnorm(1000), dim = c(lon = 3, lat = 2, sdate = 60, member = 10))
ref <- array(rnorm(1000), dim = c(lon = 3, lat = 2, sdate = 60, member = 10))
obs <- array(rnorm(1000), dim = c(lon = 3, lat = 2, sdate = 60))
ROCSS(exp = exp, obs = obs) ## random forecast as reference forecast
ROCSS(exp = exp, obs = obs, ref = ref) ## ref as reference forecast
# Use probs as input
exp_probs <- GetProbs(exp, memb_dim = 'member')
obs_probs <- GetProbs(obs, memb_dim = NULL)
ref_probs <- GetProbs(ref, memb_dim = 'member')
ROCSS(exp = exp_probs, obs = obs_probs, ref = ref_probs, memb_dim = NULL, cat_dim = 'bin')

}
\references{
Kharin, V. V. and Zwiers, F. W. (2003): 
 https://doi.org/10.1175/1520-0442(2003)016%3C4145:OTRSOP%3E2.0.CO;2
}
