#' @title Encrypt a string
#'
#' @description \code{encrypt_string} encrypts a string as a string and
#'   \code{decrypt_string} decrypts the encrypted string(encrypted using
#'   \code{encrypt_string})
#'
#' @param string A string(character vector of length 1) without embedded NULL to
#'   be encrypted
#' @param key A string without embbeded NULL. Default is 'pass'
#' @param method Currently, a stub. It should be 'symmetric'(default)
#'
#' @return An encrypted string
#'
#' @examples
#' temp <- encrypt_string("hello, how are you", key = "secret")
#' all(
#'   is.character(temp)
#'   , decrypt_string(temp, "secret") == "hello, how are you"
#'   , class(try(decrypt_string(temp, "nopass"), silent = TRUE)) == "try-error")
#'
#' @export

encrypt_string <- function(string
                           , key = "pass"
                           , method = "symmetric"){

  assert_that(is.string(string))
  assert_that(is.string(key))
  assert_that(method %in% c("symmetric"))

  stringAsRaw <- try(charToRaw(string), silent = TRUE)
  if(is.error(stringAsRaw)){
    stop("Unable to convert string to raw. Ensure string does not have a embedded NULL")
  }
  keyAsRaw <- try(hash(charToRaw(key)), silent = TRUE)
  if(is.error(keyAsRaw)){
    stop("Unable to convert 'key' into raw. Possibly encountered an embedded NULL.")
  }

  if(method == "symmetric"){
    string_enc_raw <-
      data_encrypt(stringAsRaw
                   , keyAsRaw
                   , hash(charToRaw("nounce"), size = 24)
      )
  }

  string_enc <- base64encode(string_enc_raw)
  return(string_enc)
}

#' @title Decrypt an encrypted string
#'
#' @description \code{encrypt_string} encrypts a string as a string and
#'   \code{decrypt_string} decrypts the encrypted string(encrypted using
#'   \code{encrypt_string})
#'
#' @param encryptedString A string(character vector of length 1) without embedded NULL to be decrypted.
#' @param key A string without embbeded NULL. Default is 'pass'.
#' @param method Currently, a stub. It should be 'symmetric'(default)
#'
#' @return A decrypted string
#'
#' @examples
#' temp <- encrypt_string("hello, how are you", key = "secret")
#' all(
#'   is.character(temp)
#'   , decrypt_string(temp, "secret") == "hello, how are you"
#'   , class(try(decrypt_string(temp, "nopass"), silent = TRUE)) == "try-error")
#'
#' @export

decrypt_string <- function(encryptedString
                           , key = "pass"
                           , method = "symmetric"){

  assert_that(is.string(encryptedString))
  assert_that(is.string(key))
  assert_that(method %in% c("symmetric"))

  stringAsRaw <- try(base64decode(encryptedString), silent = TRUE)
  if(is.error(stringAsRaw)){
    stop("Unable to decode encryptedString. Ensure that the input was generated by 'encrypt_string' function")
  }
  keyAsRaw <- try(hash(charToRaw(key)), silent = TRUE)
  if(is.error(keyAsRaw)){
    stop("Unable to convert 'key' into raw. Possibly encountered an embedded NULL.")
  }

  if(method == "symmetric"){
    string_dec_raw <-
      try(data_decrypt(stringAsRaw
                       , keyAsRaw
                       , hash(charToRaw("nounce"), size = 24))
          , silent = TRUE)
  }
  if(is.error(string_dec_raw)){
    stop("Unable to decrypt. Ensure that the input was generated by 'encrypt_string'. Ensure that 'key' and 'method' are correct.")
  }
  return(rawToChar(string_dec_raw))
}
