% Generated by roxygen2 (4.0.2): do not edit by hand
\name{saps}
\alias{saps}
\title{Compute SAPS statistics}
\usage{
saps(candidateGeneSets, dataSet, survivalTimes, followup,
  random.samples = 1000, cpus = 1, gsea.perm = 1000,
  compute_qvalue = FALSE, qvalue.samples = 1000, verbose = TRUE)
}
\arguments{
\item{candidateGeneSets}{A matrix with at least one row, where each row represents
a gene set, and the column values are gene identifiers. The row names should contain
unique names for the gene sets. The column values may contain \code{NA} values, since
in general gene sets will have differing lengths.}

\item{dataSet}{A matrix, where the column names are gene identifiers
(in the same format as the values in \code{candidateGeneSets})
and the values are gene expression levels. Each row should contain data for a single
patient.}

\item{survivalTimes}{A vector of survival times. The length must equal the number of
rows (i.e. patients) in \code{dataSet}.}

\item{followup}{A vector of 0 or 1 values, indicating whether the patient was
lost to followup (0) or not (1). The length must equal the number of rows
(i.e. patients) in \code{dataSet}.}

\item{random.samples}{An integer that specifies how many random gene sets to sample
when computing P_random. Defaults to 1000.}

\item{cpus}{An integer that specifies the number of cpus/cores to be used when
calculating P_enrichment. If greater than 1 (the default), the \pkg{snowfall}
package must be installed or an error will occur.}

\item{gsea.perm}{The number of permutations to be used when calculating
p_enrich. This is passed to the \code{\link[piano]{runGSA}} function in the
\pkg{piano} package. Defaults to 1000.}

\item{compute_qvalue}{A boolean indicating whether to include calculation
of the saps q_value. Setting this to \code{TRUE} will significantly
increase the computational time.}

\item{qvalue.samples}{An integer that specifies how many random gene sets to
sample when computing the saps q_value. Defaults to 1000.}

\item{verbose}{A boolean indicating whether to display status messages during
computation. Defaults to \code{TRUE}.}
}
\value{
The function returns a list with the following elements:

\item{rankedGenes}{Vector of concordance index z-scores for the genes in
   \code{dataSet}, named by gene identifier.}
\item{geneset.count}{The number of gene sets analyzed.}
\item{genesets}{A list of genesets (see below).}
\item{saps_table}{A dataframe summarizing the adjusted and unadjusted
   saps statistics for each geneset analyzed. The dataframe contains
   the following columns: \code{size, p_pure, p_random, p_enrich,
   direction, saps_score, saps_qvalue, p_pure_adj, p_random_adj,
   p_enrich_adj, saps_score_adj, saps_qvalue_adj}. Each row summarizes
   a single geneset. Note that the saps statistics are stored
   with each individual \code{geneset} as well; this table is
   provided simply for convenience.}
\code{genesets} is in turn a list with the following elements:

\item{name}{The name of the geneset.}
\item{size}{The number of genes in the geneset.}
\item{genes}{Vector of gene labels for this geneset.}
\item{saps_unadjusted}{Vector with elements \code{p_pure}, \code{p_random},
    \code{p_enrich}, \code{saps_score}, and \code{saps_qvalue} containing
    the respective unadjusted p-values.}
\item{saps_adjusted}{Vector with elements \code{p_pure}, \code{p_random},
    \code{p_enrich}, \code{saps_score}, and \code{saps_qvalue} containing
    the respective p-values adjusted for multiple comparisons.}
\item{cluster}{Vector of assigned cluster (1 or 2) for each patient using this
    candidate geneset.}
\item{random_p_pures}{Vector of p_pure values for each random geneset generated
    during the computation of p_random.}
\item{random_saps_scores}{Vector of saps_score values for each random geneset
    generated during the computation of saps_qvalue.}
\item{direction}{Direction (-1 or 1) of the enrichment association for this geneset.}
}
\description{
This is the main user interface to the \pkg{saps} package, and is
usually the only function needed.
}
\details{
saps provides a robust method for identifying biologically significant
gene sets associated with patient survival. Three basic statistics are computed.
First, patients are clustered into two survival groups based on differential
expression of a candidate gene set. p_pure is calculated as the probability of
no survival difference between the two groups.

Next, the same procedure is applied to randomly generated gene sets, and p_random
is calculated as the proportion achieving a p_pure as significant as the candidate
gene set. Finally, a pre-ranked Gene Set Enrichment Analysis (GSEA) is performed
by ranking all genes by concordance index, and p_enrich is computed to indicate
the degree to which the candidate gene set is enriched for genes with univariate
prognostic significance.

A saps_score is calculated to summarize the three statistics, and
optionally a saps_qvalue is computed to estimate the significance of the
saps_score by calculating the saps_score for random gene sets.
}
\examples{
# 25 patients, none lost to followup
followup <- rep(1, 25)

# first 5 patients have good survival (in days)
time <- c(25, 27, 24, 21, 26, sample(1:3, 20, TRUE))*365

# create data for 100 genes, 25 patients
dat <- matrix(rnorm(25*100), nrow=25, ncol=100)
colnames(dat) <- as.character(1:100)

# create two random genesets of 5 genes each
set1 <- sample(colnames(dat), 5)
set2 <- sample(colnames(dat), 5)

genesets <- rbind(set1, set2)

# compute saps
\dontrun{
results <- saps(genesets, dat, time, followup, random.samples=100)

# check results
saps_table <- results$saps_table
saps_table[1:7]

# increase expression levels for set1 for first 5 patients
dat[1:5, set1] <- dat[1:5, set1]+10

# run again, should get significant values for set1
results <- saps(genesets, dat, time, followup, random.samples=100)

# check results
saps_table <- results$saps_table
saps_table[1:7]
}
}
\references{
Beck AH, Knoblauch NW, Hefti MM, Kaplan J, Schnitt SJ, et al.
(2013) Significance Analysis of Prognostic Signatures. PLoS Comput Biol 9(1):
e1002875.doi:10.1371/journal.pcbi.1002875
}
\seealso{
\code{\link[survival]{survdiff}} \code{\link[survcomp]{concordance.index}}
    \code{\link[piano]{runGSA}}
}

