% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fss_subset_score.R
\name{subset_aggregation_FF}
\alias{subset_aggregation_FF}
\title{Fast Subset Aggregation over both locations and data streams.}
\usage{
subset_aggregation_FF(args, score_fun = poisson_score,
  priority_fun = poisson_priority, R = 50, rel_tol = 0.01)
}
\arguments{
\item{args}{A list of matrices:
   \describe{
     \item{counts}{Required. A matrix of counts. Rows indicate time, ordered 
                   from most recent to most distant. Columns indicate e.g. 
                   locations or data streams, enumerated from 1 and up.}
     \item{baselines}{Required. A matrix of expected counts. Dimensions are 
                      as for \code{counts}.}
     \item{penalties}{Optional. A matrix of penalty terms. Dimensions are as
                      for \code{counts}.}
     \item{...}{Optional. More matrices with parameters}
}}

\item{score_fun}{A function taking matrix arguments, all of the
same dimension, and returning a matrix or vector of that dimension. 
Suitable alternatives are \code{\link{poisson_score}}, 
\code{\link{gaussian_score}}.}

\item{priority_fun}{A function taking matrix arguments, all of the
same dimension, and returning a matrix or vector of that dimension. 
Suitable alternatives are \code{\link{poisson_priority}}, 
\code{\link{gaussian_priority}}.}

\item{R}{The number of random restarts.}

\item{rel_tol}{The relative tolerance criterion. If the current score divided
by the previous score, minus one, is less than this number then the 
algorithm is deemed to have converged.}
}
\value{
A list containing the most likely cluster (MLC), having the following 
   elements:
   \describe{
     \item{score}{A scalar; the score of the MLC.}
     \item{duration}{An integer; the duration of the MLC, i.e. how many time 
                     periods from the present into the past the MLC 
                     stretches.}
     \item{locations}{An integer vector; the locations contained in the MLC.}
     \item{streams}{An integer vector; the data streams contained in the 
                    MLC.}
     \item{random_restarts}{The number of random restarts performed.}
     \item{iter_to_conv}{The number of iterations it took to reach 
                         convergence for each random restart.}
   }
}
\description{
Compute the most likely cluster (MLC) with the Subset Aggregation method by
Neill et al. (2013) through fast optimization over subsets of locations and 
subsets of streams.
}
\details{
Note: algorithm not quite as in Neill et al. (2013) since the 
   randomly chosen subset of streams is the same for all time windows.
}
\examples{
\dontrun{
# Set simulation parameters (small)
set.seed(1)
n_loc <- 20
n_dur <- 10
n_streams <- 2
n_tot <- n_loc * n_dur * n_streams

# Generate baselines and possibly other distribution parameters
baselines <- rexp(n_tot, 1/5) + rexp(n_tot, 1/5)
sigma2s <- rexp(n_tot)

# Generate counts
counts <- rpois(n_tot, baselines)

# Reshape into arrays
counts <- array(counts, c(n_dur, n_loc, n_streams))
baselines <- array(baselines, c(n_dur, n_loc, n_streams))
sigma2s <- array(sigma2s, c(n_dur, n_loc, n_streams))

# Inject an outbreak/event
ob_loc <- 1:floor(n_loc / 4)
ob_dur <- 1:floor(n_dur / 4)
ob_streams <- 1:floor(n_streams / 2)
counts[ob_dur, ob_loc, ob_streams] <- 4 * counts[ob_dur, ob_loc, ob_streams]

# Run the FN algorithm
FF_res <- subset_aggregation_FF(
  list(counts = counts, baselines = baselines),
  score_fun = poisson_score,
  priority_fun = poisson_priority,
  algorithm = "FN")
}
}
\references{
Neill, Daniel B., Edward McFowland, and Huanian Zheng (2013). \emph{Fast 
   subset scan for multivariate event detection}. Statistics in Medicine 
   32 (13), pp. 2185-2208.
}
\keyword{internal}
