% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scdata.R
\name{scdata}
\alias{scdata}
\title{Data preparation to use before calling \code{scest} or \code{scpi} for point estimation and inference procedures using Synthetic Control.}
\usage{
scdata(
  df,
  id.var,
  time.var,
  outcome.var,
  period.pre,
  period.post,
  unit.tr,
  unit.co,
  features = NULL,
  cov.adj = NULL,
  cointegrated.data = FALSE,
  anticipation = 0,
  constant = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{df}{a dataframe object.}

\item{id.var}{a character or numeric scalar with the name of the variable containing units' IDs. The ID variable can be numeric or character.}

\item{time.var}{a character with the name of the time variable. The time variable has to be numeric, integer, or Date. In
case \code{time.var} is Date it should be the output of \code{\link{as.Date}()} function. An integer or
numeric time variable is suggested when working with yearly data, whereas for all other formats a Date type
time variable is preferred.}

\item{outcome.var}{a character with the name of the outcome variable. The outcome variable has to be numeric.}

\item{period.pre}{a numeric vector that identifies the pre-treatment period in time.var.}

\item{period.post}{a numeric vector that identifies the post-treatment period in time.var.}

\item{unit.tr}{a character or numeric scalar that identifies the treated unit in \code{id.var}.}

\item{unit.co}{a character or numeric vector that identifies the donor pool in \code{id.var}.}

\item{features}{a character vector containing the name of the feature variables used for estimation.
If features is specified, then outcome.var must be included in it. If this option is not specified the
default is \code{features = outcome.var}.}

\item{cov.adj}{a list specifying the names of the covariates to be used for adjustment for each feature. See the \strong{Details} section for more.}

\item{cointegrated.data}{a logical that indicates if there is a belief that the data is cointegrated or not. The default value is \code{FALSE}.  See the \strong{Details} section for more.}

\item{anticipation}{a scalar that indicates the number of periods of potential anticipation effects. Default is 0.}

\item{constant}{a logical which controls the inclusion of a constant term across features. The default value is \code{FALSE}.}

\item{verbose}{if \code{TRUE} prints additional information in the console.}
}
\value{
The command returns an object of class 'scpi_data' containing the following
\item{A}{a matrix containing pre-treatment features of the treated unit.}
\item{B}{a matrix containing pre-treatment features of the control units.}
\item{C}{a matrix containing covariates for adjustment.}
\item{P}{a matrix whose rows are the vectors used to predict the out-of-sample series for the synthetic unit.}
\item{Y.pre}{a matrix containing the pre-treatment outcome of the treated unit.}
\item{Y.post}{a matrix containing the post-treatment outcome of the treated unit.}
\item{Y.donors}{a matrix containing the pre-treatment outcome of the control units.}
\item{specs}{a list containing some specifics of the data:
\itemize{
\item{\code{J}, the number of control units}
\item{\code{K}, a numeric vector with the number of covariates used for adjustment for each feature}
\item{\code{KM}, the total number of covariates used for adjustment}
\item{\code{M}, number of features}
\item{\code{period.pre}, a numeric vector with the pre-treatment period}
\item{\code{period.post}, a numeric vector with the post-treatment period}
\item{\code{T0.features}, a numeric vector with the number of periods used in estimation for each feature}
\item{\code{T1.outcome}, the number of post-treatment periods}
\item{\code{outcome.var}, a character with the name of the outcome variable}
\item{\code{features}, a character vector with the name of the features}
\item{\code{constant}, for internal use only}
\item{\code{out.in.features}, for internal use only}
\item{\code{effect}, for internal use only}
\item{\code{treated.units}, list containing the IDs of all treated units}}}
}
\description{
The command prepares the data to be used by \code{\link{scest}} or \code{\link{scpi}} to implement estimation and inference procedures for Synthetic Control (SC) methods.
It allows the user to specify the outcome variable, the features of the treated unit to be
matched, and covariate-adjustment feature by feature. The names of the output matrices
follow the terminology proposed in \href{https://cattaneo.princeton.edu/papers/Cattaneo-Feng-Titiunik_2021_JASA.pdf}{Cattaneo, Feng, and Titiunik (2021)}.

Companion \href{https://www.stata.com/}{Stata} and \href{https://www.python.org/}{Python} packages are described in \href{https://arxiv.org/abs/2202.05984}{Cattaneo, Feng, Palomba, and Titiunik (2022)}.

Companion commands are: \link{scdataMulti} for data preparation in the multiple treated units case with staggered adoption,
\link{scest} for point estimation, \link{scpi} for inference procedures, \link{scplot} and \link{scplotMulti} for plots in the single and multiple treated unit(s) cases, respectively.

Related Stata, R, and Python packages useful for inference in SC designs are described in the following website:

\href{https://nppackages.github.io/scpi/}{ https://nppackages.github.io/scpi/}

For an introduction to synthetic control methods, see \href{https://www.aeaweb.org/articles?id=10.1257/jel.20191450}{Abadie (2021)} and references therein.
}
\details{
\itemize{
\item{\code{cov.adj} can be used in two ways. First, if only one feature is specified through the option \code{features},
\code{cov.adj} has to be a list with one (even unnamed) element (eg. \code{cov.adj = list(c("constant","trend"))}).
Alternatively, if multiple features are specified, then the user has two possibilities:
\itemize{
\item{provide a list with one element, then the same covariates are used for
adjustment for each feature. For example, if there are two features specified and the user inputs
\code{cov.adj = list(c("constant","trend"))}, then a constant term and a linear trend are for adjustment for both features.}
\item{provide a list with as many elements as the number of features specified, then
feature-specific covariate adjustment is implemented. For example,
\code{cov.adj = list('f1' = c("constant","trend"), 'f2' = c("trend"))}. In this case the name of each element
of the list should be one (and only one) of the features specified.}}

This option allows the user to include feature-specific constant terms
}

\item{\code{cointegrated.data} allows the user to model the belief that \eqn{\mathbf{A}} and \eqn{\mathbf{B}} form a
cointegrated system. In practice, this implies that when dealing with the pseudo-true
residuals \eqn{\mathbf{u}}, the first-difference of \eqn{\mathbf{B}} are used rather than the levels.}
}
}
\examples{

data <- scpi_germany

df <- scdata(df = data, id.var = "country", time.var = "year", 
             outcome.var = "gdp", period.pre = (1960:1990), 
             period.post = (1991:2003), unit.tr = "West Germany",
             unit.co = unique(data$country)[-7], constant = TRUE,
             cointegrated.data = TRUE)

}
\references{
\itemize{
\item{\href{https://www.aeaweb.org/articles?id=10.1257/jel.20191450}{Abadie, A. (2021)}. Using synthetic controls: Feasibility, data requirements, and methodological aspects.
\emph{Journal of Economic Literature}, 59(2), 391-425.}
\item{\href{https://cattaneo.princeton.edu/papers/Cattaneo-Feng-Titiunik_2021_JASA.pdf}{Cattaneo, M. D., Feng, Y., and Titiunik, R.
(2021)}. Prediction intervals for synthetic control methods. \emph{Journal of the American Statistical Association}, 116(536), 1865-1880.}
\item{\href{https://arxiv.org/abs/2202.05984}{Cattaneo, M. D., Feng, Y., Palomba F., and Titiunik, R. (2022).}
scpi: Uncertainty Quantification for Synthetic Control Estimators, \emph{arXiv}:2202.05984.}
}
}
\seealso{
\code{\link{scdataMulti}}, \code{\link{scest}}, \code{\link{scpi}}, \code{\link{scplot}}, \code{\link{scplotMulti}}
}
\author{
Matias Cattaneo, Princeton University. \email{cattaneo@princeton.edu}.

Yingjie Feng, Tsinghua University. \email{fengyj@sem.tsinghua.edu.cn}.

Filippo Palomba, Princeton University (maintainer). \email{fpalomba@princeton.edu}.

Rocio Titiunik, Princeton University. \email{titiunik@princeton.edu}.
}
