\name{sim.popn}
\alias{sim.popn}
\alias{tile}

\title{ Simulate 2-D Population }

\description{
Simulate a Poisson process representing the locations of individual animals.
}

\usage{

sim.popn (D, core, buffer = 100, model2D = c("poisson", "cluster",
  "IHP", "coastal", "hills", "linear", "even"), buffertype = c("rect",
  "concave", "convex"), poly = NULL, covariates = list(sex = c(M = 0.5,
  F = 0.5)), number.from = 1, Ndist = c("poisson", "fixed",
  "specified"), nsessions = 1, details = NULL, seed = NULL, keep.mask =
  model2D \%in\% c("IHP", "linear"), Nbuffer = NULL, age = FALSE, ...)

tile(popn, method = "reflect")

}

\arguments{
  \item{D}{ density animals / hectare (10 000 m^2) (see Details for
  IHP case) }
  \item{core}{ data frame of points defining the core area }
  \item{buffer}{ buffer radius about core area }
  \item{model2D}{ character string for 2-D distribution }
  \item{buffertype}{ character string for buffer type }
  \item{poly}{ bounding polygon (see Details)}
  \item{covariates}{ list of named covariates }
  \item{number.from}{ integer ID for animal }
  \item{Ndist}{ character string for distribution of number of individuals }
  \item{nsessions}{ number of sessions to simulate }
  \item{details}{ optional list with additional parameters }
  \item{seed}{either NULL or an integer that will be used in a call to \code{set.seed}}
  \item{keep.mask}{logical; if TRUE and model2D \%in\% c('IHP','linear')
  then \code{core} is saved as the attribute "mask"}
  \item{Nbuffer}{integer number of individuals to simulate}
  \item{age}{logical; if TRUE then age covariate added for multisession popn with turnover}
  \item{...}{ arguments passed to subset if poly is not NULL}
  \item{popn}{ popn object }
  \item{method}{ character string "reflect" or "copy"}
}
\details{

\code{core} must contain columns `x' and `y'; a \code{traps} object is
suitable. For \code{buffertype = "rect"}, animals are simulated in the
rectangular area obtained by extending the bounding box of \code{core}
by \code{buffer} metres to top and bottom, left and right. This box has
area \eqn{A}. If \code{model2D = 'poisson'} the buffer type may also be `convex' (points within a buffered convex polygon) or `concave' (corresponding to a mask of type `trapbuffer'); these buffer types use \code{\link{buffer.contour}}.

A notional random covariate `sex' is generated by default.

Each element of \code{covariates} defines a categorical (factor)
covariate with the given probabilities of membership in each class. No
mechanism is provided for generating continuous covariates, but these
may be added later (see Examples).

Ndist should usually be `poisson' or `fixed'. The number of individuals \eqn{N} has
expected value \eqn{DA}. If \eqn{DA} is non-integer then Ndist = "fixed"
results in \eqn{N \in \{ \mathrm{trunc}(DA), \mathrm{trunc}(DA)+1 \} }{N
in \{ trunc(DA), trunc(DA)+1 \} }, with probabilities set to yield
\eqn{DA} individuals on average. The option `specified' is undocumented;
it is used in some open-population simulations.

If \code{model2D = "cluster"} then the simulated population approximates a Neyman-Scott
clustered Poisson distribution. Ancillary parameters are passed as
components of \code{details}: details$mu is the expected number of
individuals per cluster and details$hsigma is the spatial scale
(\eqn{\sigma}{sigma}) of a 2-D kernel for location within each cluster.
The algorithm is
\enumerate{
\item Determine the number of clusters (parents) as a random Poisson variate
with \eqn{\lambda = DA/\mu}{ lambda = DA/mu }
\item Locate each parent by drawing uniform random x- and y-coordinates
\item Determine number of offspring for each parent by drawing from a Poisson distribution with mean mu
\item Locate offspring by adding random normal error to each parent coordinate
\item Apply toroidal wrapping to ensure all offspring locations are
inside the buffered area
}

Toroidal wrapping is a compromise. The result is more faithful to the
Neyman-Scott distribution if the buffer is large enough that only a
small proportion of the points are wrapped.

If \code{model2D = "IHP"} then an inhomogeneous Poisson distribution is
simulated.  \code{core} should be a habitat \link{mask} and \code{D}
should be one of --
\itemize{
\item a vector of length equal to the number of cells (rows)
in \code{core}, 
\item the name of a covariate in \code{core} that contains
cell-specific densities (animals / hectare),
\item a function to generate the intensity of the distribution at each mask point, or
\item a constant. 
}

If a function, \code{D} should take two arguments, a habitat mask and a list of parameter values ('core' and 'details' are passed internally as these arguments).

The number
of individuals in each cell is either (i) Poisson-distributed with mean
\eqn{DA} where \eqn{A} is the cell area (an attribute of the mask)
(\code{Ndist = "poisson"}) or (ii) multinomial with size \eqn{DA} and
relative cell probabilities given by D (\code{Ndist =
"fixed"}). \code{buffertype} and \code{buffer} are ignored, as the
extent of the population is governed entirely by the mask in
\code{core}.

If \code{model2D = "linear"} then a linear population is simulated as
for \code{model2D = "IHP"}, except that \code{core} should be a
linearmask object from package \pkg{secrlinear}, and density (D) is
expressed in animals per km. The documentation of \pkg{secrlinear}
should be consulted for further detail (e.g. the wrapper function
\code{sim.linearpopn}).

If \code{model2D = "coastal"} then a form of inhomogeneous Poisson
distribution is simulated in which the x- and y-coordinates are drawn from
independent Beta distributions. Default parameters generate the
`coastal' distribution used by Fewster and Buckland (2004) for
simulations of line-transect distance sampling (x ~ Beta(1, 1.5), y ~
Beta(5, 1), which places 50\% of the population in the `northern' 13\%
of the rectangle). The four Beta parameters may be supplied in the
vector component Beta of the `details' list (see Examples). The Beta
parameters (1,1) give a uniform distribution. Coordinates are scaled to
fit the limits of a sampled rectangle, so this method assumes buffertype
= "rect".

If \code{model2D = "hills"} then a form of inhomogeneous Poisson
distribution is simulated in which intensity is a sine curve in the x-
and y- directions (density varies symmetrically between 0 and 2 x D
along each axis). The number of hills in each direction (default 1) is
determined by the `hills' component of the `details' list (e.g. details
= list(hills=c(2,3)) for 6 hills). If either number is negative then
alternate rows will be offset by half a hill. Displacements of the
entire pattern to the right and top are indicated by further elements of
the `hills' component (e.g. details = list(hills=c(1,1,0.5,0.5)) for 1
hill shifted half a unit to the top right; coordinates are wrapped, so
the effect is to split the hill into the four corners). Negative
displacements are replaced by runif(1). Density is zero at the edge when
the displacement vector is (0,0) and rows are not offset.

If \code{model2D = "even"} then the buffered area is divided into square cells with side sqrt(10000/D) and one animal is located at a random uniform location within each cell. If the height or width is not an exact multiple of the cell side then one whole extra row or column of cells is added; animals located at random in these cells are discarded if they fall outside the original area.

If \code{poly} is specified, points outside \code{poly} are
dropped. \code{poly} may be one of the types descrbed in 
\code{\link{boundarytoSF}}.

The \code{\link{subset}} method is called internally when \code{poly} is used;
the \dots argument may be used to pass values for \code{keep.poly} and
\code{poly.habitat}.

Multi-session populations may be generated with \code{nsessions > 1}. 
Multi-session populations may be independent or generated by per capita 
turnover from a starting population. In the `independent' case
(\code{details$lambda} not specified) D or Nbuffer may be a vector of length equal to
\code{nsessions}. Turnover is controlled by survival, growth rate and movement
parameters provided as components of \code{details} and described in \link{turnover}. 
The optional covariate 'age' is the number of sessions from the session of recruitment.

The random number seed is managed as in \code{simulate.lm}.


Function \code{tile} replicates a popn pattern by either reflecting or
copying and translating it to fill a 3 x 3 grid.

}

\value{

An object of class \code{c("popn", "data.frame")} a data frame with columns `x' and `y'. Rows correspond to individuals. Individual covariates (optional) are stored
as a data frame attribute. The initial state of the R random number generator is
stored in the `seed' attribute.

If \code{model2D = "linear"} the output is of class \code{c("linearpopn",
  "popn", "data.frame")}.

If \code{model2D = "IHP"} or \code{model2D = "linear"} the value of
\code{core} is stored in the `mask' attribute.

}

\references{

  Fewster, R. M. and Buckland, S. T. 2004. Assessment of distance
  sampling estimators. In: S. T. Buckland, D. R. Anderson,
  K. P. Burnham, J. L. Laake, D. L. Borchers and L. Thomas (eds)
  \emph{Advanced distance sampling}. Oxford University Press, Oxford,
  U. K. Pp. 281--306.
  
}

\seealso{ \code{\link{popn}}, \code{\link{plot.popn}},
  \code{\link{randomHabitat}}, \link{turnover}, 
  \code{\link[stats]{simulate}} }

\examples{

temppop <- sim.popn (D = 10, expand.grid(x = c(0,100), y =
    c(0,100)), buffer = 50)

## plot, distinguishing "M" and "F"
plot(temppop, pch = 1, cex= 1.5,
    col = c("green","red")[covariates(temppop)$sex])

## add a continuous covariate
## assumes covariates(temppop) is non-null
covariates(temppop)$size <- rnorm (nrow(temppop), mean = 15, sd = 3)
summary(covariates(temppop))

## Neyman-Scott cluster distribution
par(xpd = TRUE, mfrow=c(2,3))
for (h in c(5,15))
for (m in c(1,4,16)) {
    temppop <- sim.popn (D = 10, expand.grid(x = c(0,100),
        y = c(0,100)), model2D = "cluster", buffer = 100,
        details = list(mu = m, hsigma = h))
    plot(temppop)
    text (50,230,paste(" mu =",m, "hsigma =",h))
}
par(xpd = FALSE, mfrow=c(1,1))   ## defaults

## Inhomogeneous Poisson distribution
xy <- secrdemo.0$mask$x + secrdemo.0$mask$y - 900
tempD <- xy^2 / 1000
plot(sim.popn(tempD, secrdemo.0$mask, model2D = "IHP"))

## Coastal distribution in 1000-m square, homogeneous in
## x-direction
arena <- data.frame(x = c(0, 1000, 1000, 0),
    y = c(0, 0, 1000, 1000))
plot(sim.popn(D = 5, core = arena, buffer = 0, model2D =
    "coastal", details = list(Beta = c(1, 1, 5, 1))))

## Hills
plot(sim.popn(D = 100, core = arena, model2D = "hills",
    buffer = 0, details = list(hills = c(-2,3,0,0))), 
    cex = 0.4)

## tile demonstration
pop <- sim.popn(D = 100, core = make.grid(), model2D = "coastal")
par(mfrow = c(1,2), mar = c(2,2,2,2))
plot(tile(pop, "copy"))
polygon(cbind(-100,200,200,-100), c(-100,-100,200,200),
    col = "red", density = 0)
title("copy")
plot(tile(pop, "reflect"))
polygon(cbind(-100,200,200,-100), c(-100,-100,200,200),
    col = "red", density = 0)
title("reflect")


\dontrun{
## simulate from inhomogeneous fitted density model

regionmask <- make.mask(traps(possumCH), type = "polygon",
    spacing = 20, poly = possumremovalarea)
dts <- distancetotrap(regionmask, possumarea)
covariates(regionmask) <- data.frame(d.to.shore = dts)
dsurf <- predictDsurface(possum.model.Ds, regionmask)
possD <- covariates(dsurf)$D.0
posspop <- sim.popn(D = possD, core = dsurf, model = "IHP")
plot(regionmask, dots = FALSE, ppoly = FALSE)
plot(posspop, add = TRUE, frame = FALSE)
plot(traps(possumCH), add = TRUE)
}

}



\keyword{ datagen }

