% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_boot.R
\name{hist_qq_boot}
\alias{hist_qq_boot}
\alias{scatter_boot}
\title{Diagnostic Plots of Bootstrap Estimates in 'lavaan'}
\usage{
hist_qq_boot(
  object,
  param,
  standardized = NULL,
  nclass = NULL,
  hist_color = "#5DADE233",
  hist_linewidth = 1.5,
  hist_border_color = "#1B4F72",
  density_line_type = "solid",
  density_line_color = "#8B0000CC",
  density_line_linewidth = 2,
  est_line_color = "#154360",
  est_line_type = "dashed",
  est_line_linewidth = 2,
  qq_dot_pch = 21,
  qq_dot_color = "#1B4F72",
  qq_dot_fill = "#5DADE233",
  qq_dot_size = 1.3,
  qq_line_color = "#8B0000CC",
  qq_line_linewidth = 2.1,
  qq_line_linetype = "solid"
)

scatter_boot(
  object,
  params,
  standardized = NULL,
  main = "Bootstrap Estimates",
  ...
)
}
\arguments{
\item{object}{Either
a \code{lavaan-class}
object with bootstrap estimates
stored, or the output of
\code{\link[=standardizedSolution_boot]{standardizedSolution_boot()}}.
For standardized solution
and user-defined parameters, if
the object is a `lavaan-class``
object, the
estimates need to be stored by
\code{\link[=store_boot]{store_boot()}}.}

\item{param}{String. The name of
the parameter to be plotted, which
should be the name as appeared in
a call to \code{coef()}.}

\item{standardized}{Logical. Whether
the estimates from the standardized
solution are to be plotted. Default
is \code{NULL}. If \code{object} is a
\code{lavaan} object, then
this is a required parameter
and users need to explicitly set it
to \code{TRUE} or \code{FALSE}. If \code{object} is
the output of
\code{\link[=standardizedSolution_boot]{standardizedSolution_boot()}},
then this argument is ignored (
forced to be \code{TRUE} internally).}

\item{nclass}{The number of breaks.
This argument will be passed to
\code{\link[=hist]{hist()}}. Default is \code{NULL}.}

\item{hist_color}{String. The color of the
bars in the histogram. It will be
passed to \code{\link[=hist]{hist()}} for the argument
\code{col}. Default is light blue (\code{scales::alpha("#5DADE2", 0.2)}).}

\item{hist_linewidth}{The width of
the borders of the bars in the
histogram. Default is 1.5.}

\item{hist_border_color}{String.
The color of the borders (outline) of the bars
in the histogram. It will be passed to \code{\link[=hist]{hist()}}
for the argument \code{border}.
Default is a dark blue color (\code{"#1B4F72"}).}

\item{density_line_type}{String.
The type of the line of the density
curve in the histogram. It will be
passed to \code{\link[=lines]{lines()}} for the argument
\code{lty}. Default is
\code{"solid"}.}

\item{density_line_color}{String.
The color of the density curve in
the histogram. It will be
passed to \code{\link[=lines]{lines()}} for the argument
\code{col}. Default is \code{"blue"}.}

\item{density_line_linewidth}{The width
of the density curve in the histogram.
It will be
passed to \code{\link[=lines]{lines()}} for the argument
\code{lwd}.
Default is 2.}

\item{est_line_color}{String. The
color of the vertical line showing
the point estimate in the histogram.
It will be
passed to \code{\link[=abline]{abline()}} for the argument
\code{col}.}

\item{est_line_type}{String. The
type of the vertical line in the
histogram showing the point estimate
of the parameter. It will be
passed to \code{\link[=abline]{abline()}} for the argument
\code{lty}. Default is
\code{"dashed"},}

\item{est_line_linewidth}{The width
of the vertical line showing the
point estimate in the histogram.
It will be
passed to \code{\link[=hist]{hist()}} for the argument
\code{lwd}.  Default is 2.}

\item{qq_dot_pch}{Numeric. The shape
of the points in the normal QQ-plot.
It will be
passed to \code{\link[=qqnorm]{qqnorm()}} for the argument
\code{pch}. Default is 21.}

\item{qq_dot_color}{String. The color
of the points in the normal QQ-plot.
It will be
passed to \code{\link[=qqnorm]{qqnorm()}} for the argument
\code{col}.}

\item{qq_dot_fill}{String.
The fill color of the points in the normal QQ-plot.
Only applicable when \code{qq_dot_pch} is set to a symbol
that allows fill color (e.g., \code{pch = 21}).
It will be passed to \code{\link[=qqnorm]{qqnorm()}} for the argument \code{bg}.
Default is a semi-transparent light blue
(\code{scales::alpha("#5DADE2", 0.2)}).}

\item{qq_dot_size}{The size of the
points in the normal QQ-plot.
It will be
passed to \code{\link[=qqnorm]{qqnorm()}} for the argument
\code{cex}. Default is 2.}

\item{qq_line_color}{String. The color
of the diagonal line to be drawn in
the normal QQ-plot.
It will be
passed to \code{\link[=qqline]{qqline()}} for the argument
\code{col}.}

\item{qq_line_linewidth}{The width
of the diagonal line to be drawn in
the normal QQ-plot.
It will be
passed to \code{\link[=qqline]{qqline()}} for the argument
\code{lwd}. Default is 2.1.}

\item{qq_line_linetype}{The type of
the diagonal line to be drawn in the
normal QQ-plot. Default is \code{"solid"}.}

\item{params}{The vector of the names of
the parameters to be plotted, which
should be the names as appeared in
a call to \code{coef()}. The function
\code{\link[=scatter_boot]{scatter_boot()}} requires two or more
parameters selected by this argument.}

\item{main}{The title of the
scatterplot matrix. Default is
\code{"Bootstrap Estimates"}.}

\item{...}{Arguments to be passed to
\code{\link[psych:pairs.panels]{psych::pairs.panels()}}. Please refer
to the help page of \code{\link[psych:pairs.panels]{psych::pairs.panels()}}
for arguments to customize the
plot.}
}
\value{
Return the original
\link[lavaan:lavaan-class]{lavaan::lavaan} object
invisibly. Called for its side-effect
(plotting the graphs).
}
\description{
Plots for examining the
distribution of bootstrap estimates
in a model fitted by \code{lavaan}.
}
\details{
Rousselet, Pernet, and Wilcox (2021)
argued that when using bootstrapping,
it is necessary to examine the distribution
of bootstrap estimates. This can be
done when \code{\link[boot:boot]{boot::boot()}} is used
because it has a \code{plot} method for
its output. This cannot be easily
done in model fitted by \code{\link[lavaan:lavaan]{lavaan::lavaan()}},
such as \code{\link[lavaan:sem]{lavaan::sem()}} and
\code{\link[lavaan:cfa]{lavaan::cfa()}}.

The function \code{\link[=hist_qq_boot]{hist_qq_boot()}} is used for
plotting the distribution of bootstrap
estimates for a model fitted by
\code{lavaan} in a format similar to that
of the output of \code{\link[boot:boot]{boot::boot()}}, with
a histogram on the left and a normal
QQ-plot on the right.

For free parameters in a model
(unstandardized), it can be called
directly on the output of \code{lavaan}
and retrieves the stored estimates.

For estimates of user-defined parameters,
call \code{\link[=store_boot]{store_boot()}} first to compute
and store the bootstrap estimates
first.

For estimates in standardized solution,
for both free and user-defined
parameters, call \code{\link[=store_boot]{store_boot()}}
first to compute and store the bootstrap
estimates in the standardized solution.

It can also
plot bootstrap estimates in the output
of \code{\link[=standardizedSolution_boot]{standardizedSolution_boot()}}
or \code{\link[=parameterEstimates_boot]{parameterEstimates_boot()}}.

The function \code{\link[=scatter_boot]{scatter_boot()}} is
used to generate a scatterplot
matrix of the bootstrap estimates of
two or more parameters. The function
\code{\link[psych:pairs.panels]{psych::pairs.panels()}} from the
package \code{psych} is used.

Like \code{\link[=hist_qq_boot]{hist_qq_boot()}}, it can also
be used on the output
of \code{\link[=standardizedSolution_boot]{standardizedSolution_boot()}}
or \code{\link[=parameterEstimates_boot]{parameterEstimates_boot()}}.
}
\examples{

library(lavaan)

set.seed(5478374)
n <- 50
x <- runif(n) - .5
m <- .40 * x + rnorm(n, 0, sqrt(1 - .40))
y <- .30 * m + rnorm(n, 0, sqrt(1 - .30))
dat <- data.frame(x = x, y = y, m = m)

mod <-
"
m ~ a * x
y ~ b * m + x
ab := a * b
"
fit <- sem(mod,
           data = dat,
           se = "bootstrap",
           bootstrap = 50,
           iseed = 985714)

# Can plot bootstrap estimates for
# free parameters directly
# Note that 'standardized' must be always be set to
# either TRUE or FALSE. No default value.
hist_qq_boot(fit, "a", standardized = FALSE)

# For estimates of user-defined parameters,
# call store_boot() first.
fit <- store_boot(fit)
hist_qq_boot(fit, "ab", standardized = FALSE)

# For estimates in standardized solution,
# call store_boot() first.
fit <- store_boot(fit)
hist_qq_boot(fit, "a", standardized = TRUE)
hist_qq_boot(fit, "ab", standardized = TRUE)

# It can also plot the estimates stored
# in the output of standardizedSolution_boot().
std_boot <- standardizedSolution_boot(fit)
hist_qq_boot(std_boot, "ab")
hist_qq_boot(fit, "ab", standardized = TRUE)



# Scatterplot matrix of bootstrap estimates for
# two or more free parameters
scatter_boot(fit, c("a", "b", "ab"), standardized = FALSE)

# Can include user-defined parameters in
# scatterplot matrix, if their bootstrap
# estimates have been stored
scatter_boot(fit, c("ab", "a", "b"), standardized = FALSE)

# scatter_boot also supports the
# standardized solution
scatter_boot(fit, c("a", "b", "ab"), standardized = TRUE)

}
\references{
Rousselet, G. A., Pernet, C. R., & Wilcox, R. R. (2021).
The percentile bootstrap: A primer with step-by-step
instructions in R.
\emph{Advances in Methods and Practices in Psychological Science},
\emph{4}(1), 1--10. \doi{10.1177/2515245920911881}
}
\seealso{
\code{\link[=store_boot]{store_boot()}}
and \code{\link[=standardizedSolution_boot]{standardizedSolution_boot()}}.
}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}
}
