% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clade_Discrete.R
\name{clade_discrete}
\alias{clade_discrete}
\title{Influential Clade Detection - Trait Evolution Discrete Characters}
\usage{
clade_discrete(data, phy, model, transform = "none", trait.col, clade.col,
  n.species = 5, n.sim = 20, bounds = list(), n.cores = NULL,
  track = TRUE, ...)
}
\arguments{
\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{model}{The Mkn model to use (see Details).}

\item{transform}{The evolutionary model to transform the tree (see Details). Default is \code{none}.}

\item{trait.col}{The column in the provided data frame which specifies the
trait to analyse (which should be a factor with two level)}

\item{clade.col}{The column in the provided data frame which specifies the
clades (a character vector with clade names).}

\item{n.species}{Minimum number of species in a clade for the clade to be
included in the leave-one-out deletion analysis. Default is \code{5}.}

\item{n.sim}{Number of simulations for the randomization test.}

\item{bounds}{settings to constrain parameter estimates. See \code{\link[geiger]{fitDiscrete}}}

\item{n.cores}{number of cores to use. If 'NULL', number of cores is detected.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[geiger]{fitDiscrete}}}
}
\value{
The function \code{tree_discrete} returns a list with the following
components:

\code{call}: The function call

\code{data}: The original full data frame.

\code{full.model.estimates}: Parameter estimates (transition rates q12 and q21), 
AICc and the optimised value of the phylogenetic transformation parameter (e.g. \code{lambda}) 
for the full model without deleted clades.

\code{sensi.estimates}: Parameter estimates (transition rates q12 and q21),(percentual) difference 
in parameter estimate compared to the full model (DIFq12, sigsq.q12, DIFq21, optpar.q21), 
AICc and the optimised value of the phylogenetic transformation parameter (e.g. \code{lambda}) 
for each repeat with a clade removed.

\code{null.dist}: A data frame with estimates for the null distributions
for all clades analysed.

\code{errors}: Clades where deletion resulted in errors.

\code{clade.col}: Which column was used to specify the clades?

\code{optpar}: Transformation parameter used (e.g. \code{lambda}, \code{kappa} etc.)
}
\description{
Fits models for trait evolution of discrete (binary) characters, 
detecting influential clades
}
\details{
This function sequentially removes one clade at a time,
fits a model of discrete character evolution using \code{\link[geiger]{fitDiscrete}}, 
repeats this this many times (controlled by \code{n.sim}), stores the results and calculates 
the effects on model parameters. Currently, only binary discrete traits are supported. 

Additionally, to account for the influence of the number of species on each 
clade (clade sample size), this function also estimates a null distribution
expected for the number of species in a given clade. This is done by fitting
 models without the same number of species as in the given clade.The number of 
 simulations to be performed is set by 'n.sim'. To test if the 
 clade influence differs from the null expectation for a clade of that size, 
 a randomization test can be performed using 'summary(x)'. 

Different character model from \code{fitDiscrete} can be used, including \code{ER} (equal-rates), 
\code{SYM} (symmetric), \code{ARD} (all-rates-different) and \code{meristic} (stepwise fashion). 

All transformations to the phylogenetic tree from \code{fitDiscrete} can be used, i.e. \code{none},
\code{EB}, \code{lambda}, \code{kappa} and\code{delta}.

See \code{\link[geiger]{fitDiscrete}} for more details on character models and tree transformations. 

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
#Load data:
data("primates")
#Create a binary trait factor 
primates$data$adultMass_binary<-ifelse(primates$data$adultMass > 7350, "big", "small")
clade_disc<-clade_discrete(data=primates$data,phy = primates$phy[[1]],model="SYM",
trait.col = "adultMass_binary",clade.col="family",n.sim=30,n.species=10,n.cores = 2)
summary(clade_disc)
sensi_plot(clade_disc)
sensi_plot(clade_disc, clade = "Cebidae", graph = "q12")
#Change the evolutionary model, tree transformation or minimum number of species per clade
clade_disc_2<-clade_discrete(data=primates$data,phy = primates$phy[[1]],
model="ARD",transform="kappa",
trait.col = "adultMass_binary",clade.col="family",n.sim=30,
n.species=8,n.cores = 2)
summary(clade_disc_2)
sensi_plot(clade_disc_2)
sensi_plot(clade_disc_2, graph = "q12")
sensi_plot(clade_disc_2, graph = "q21")
}
}
\references{
Yang Z. 2006. Computational Molecular Evolution. Oxford University Press: Oxford. 

Harmon Luke J, Jason T Weir, Chad D Brock, Richard E Glor, and Wendell Challenger. 2008.
GEIGER: investigating evolutionary radiations. Bioinformatics 24:129-131.
}
\seealso{
\code{\link[geiger]{fitDiscrete}}
}
\author{
Gijsbert Werner & Gustavo Paterno
}
