% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ComparePairs.R
\name{ComparePairs}
\alias{ComparePairs}
\title{Compare Pairwise Relationships}
\usage{
ComparePairs(
  Ped1 = NULL,
  Ped2 = NULL,
  Pairs2 = NULL,
  GenBack = 1,
  patmat = FALSE,
  ExcludeDummies = TRUE,
  DumPrefix = c("F0", "M0"),
  Return = "Counts"
)
}
\arguments{
\item{Ped1}{first (e.g. original/reference) pedigree, dataframe with 3
columns: id-dam-sire.}

\item{Ped2}{optional second (e.g. inferred) pedigree.}

\item{Pairs2}{optional dataframe with relationships categories between pairs
of individuals, instead of or in addition to \code{Ped2}, e.g. as returned
by \code{\link{GetMaybeRel}}. First three columns: ID1-ID2-relationship,
column names and any additional columns are ignored.}

\item{GenBack}{number of generations back to consider; 1 returns
parent-offspring and sibling relationships, 2 also returns grandparental,
avuncular and first cousins. GenBack >2 is not implemented.}

\item{patmat}{logical, distinguish between paternal versus maternal relative
pairs?}

\item{ExcludeDummies}{logical, exclude dummy IDs from output? Individuals
with e.g. the same dummy father will still be counted as paternal halfsibs.
No attempt is made to match dummies in one pedigree to individuals in the
other pedigree; for that use \code{\link{PedCompare}}.}

\item{DumPrefix}{character vector with the prefixes identifying dummy
individuals. Use 'F0' ('M0') to avoid matching to regular individuals with
IDs starting with 'F' ('M'), provided \code{Ped2} has fewer than 999 dummy
females (males).}

\item{Return}{return a matrix with \code{Counts} or a \code{Summary} of the
number of identical relationships and mismatches per relationship, or
detailed results as a 2xNxN \code{Array} or as a \code{Dataframe}.
\code{All} returns a list with all four.}
}
\value{
Depending on \code{Return}, one of the following, or a list with all:
\item{Counts}{(the default), a matrix with counts, with the classification in
  \code{Ped1} on rows and that in \code{Ped2} in columns. Counts for
  'symmetrical' pairs ("FS", "HS", "MHS", "PHS", "FC1", "DFC1", "U","X") are
  divided by two.}
  \item{Summary}{a matrix with one row per relationship type and four columns
    , named as if \code{Ped1} is the true pedigree:
    \describe{
      \item{n}{total number of pairs with that relationship in \code{Ped1},
        and occurring in \code{Ped2}}
     \item{OK}{Number of pairs with same relationship in \code{Ped2} as in
       \code{Ped1}}
     \item{hi}{Number of pairs with 'higher' relationship in \code{Ped2} as
       in \code{Ped1} (e.g. FS instead of HS; ranking is the order given
       below)}
     \item{lo}{Number of pairs with 'lower' relationship in \code{Ped2} as in
      \code{Ped1}, but not unrelated in \code{Ped2}}
  }}
  \item{Array}{a 2xNxN array (if \code{Ped2} or \code{Pairs2} is specified)
    or a NxN matrix , where N is the total number of individuals occurring in
    \code{Ped1} and/or \code{Ped2}.}
  \item{Dataframe}{a dataframe with \eqn{N^2} rows and four columns:
    \describe{
      \item{id.A}{First individual of the pair}
      \item{id.B}{Second individual of the pair}
      \item{RC1}{the relationship category in \code{Ped1}, as a factor with all
        considered categories as levels, including those with 0 count}
      \item{RC2}{the relationship category in \code{Ped2}}
    }
    Each pair is listed twice, e.g. once as P and once as O, or twice as FS.}
}
\description{
Compare, count and identify different types of relative pairs
  between two pedigrees, or within one pedigree.
}
\details{
If \code{Pairs2} is as returned by \code{\link{GetMaybeRel}}
  (identified by the additional column names 'LLR' and 'OH'), these
  relationship categories are appended with an '?' in the output, to
  distinguish them from those derived from \code{Ped2}.

  When \code{Pairs2$TopRel} contains values other than the ones listed among
  the return values for the combination of \code{patmat} and \code{GenBack},
  they are prioritised in decreasing order of factor levels, or in decreasing
  alphabetical order, and before the default (\code{ped2} derived) levels.

  The matrix returned by \code{\link{DyadCompare}} [Deprecated] is a subset
  of the matrix returned here using default settings.
}
\section{Relationship abbreviations and ranking}{

By default (\code{GenBack=1, patmat=FALSE}) the following 7 relationships are
distinguished:
\itemize{
   \item \strong{S}: Self (not included in \code{Counts})
   \item \strong{MP}: Parent
   \item \strong{O}: Offspring (not included in \code{Counts})
   \item \strong{FS}: Full sibling
   \item \strong{HS}: Half sibling
   \item \strong{U}: Unrelated, or otherwise related
   \item \strong{X}: Either or both individuals not occurring in both
     pedigrees
}
In the array and dataframe, 'MP' indicates that the second (column)
individual is the parent of the first (row) individual, and 'O' indicates the
reverse.

When \code{GenBack=1, patmat=TRUE} the categories are (S)-M-P-(O)-FS-MHS-PHS-
U-X.

When \code{GenBack=2, patmat=TRUE}, the following relationships are
distinguished:
\itemize{
   \item \strong{S}: Self (not included in \code{Counts})
   \item \strong{M}: Mother
   \item \strong{P}: Father
   \item \strong{O}: Offspring (not included in \code{Counts})
   \item \strong{FS}: Full sibling
   \item \strong{MHS}: Maternal half-sibling
   \item \strong{PHS}: Paternal half-sibling
   \item \strong{MGM}: Maternal grandmother
   \item \strong{MGF}: Maternal grandfather
   \item \strong{PGM}: Paternal grandmother
   \item \strong{PGF}: Paternal grandfather
   \item \strong{GO}: Grand-offspring (not included in \code{Counts})
   \item \strong{FA}: Full avuncular; maternal or paternal aunt or uncle
   \item \strong{HA}: Half avuncular
   \item \strong{FN}: Full nephew/niece (not included in \code{Counts})
   \item \strong{HN}: Half nephew/niece (not included in \code{Counts})
   \item \strong{FC1}: Full first cousin
   \item \strong{DFC1}: Double full first cousin
   \item \strong{U}: Unrelated, or otherwise related
   \item \strong{X}: Either or both individuals not occurring in both pedigrees
}
Note that for avuncular and cousin relationships no distinction is made
between paternal versus maternal, as this may differ between the two
individuals and would generate a large number of sub-classes. When a pair is
related via multiple paths, the first-listed relationship is returned.

When \code{GenBack=2, patmat=FALSE}, MGM, MGF, PGM and PGF are combined
into GP, with the rest of the categories analogous to the above.
}

\examples{
\donttest{
data(Ped_griffin, SeqOUT_griffin, package="sequoia")
PairsG <- ComparePairs(Ped_griffin, SeqOUT_griffin[["Pedigree"]],
                       patmat = TRUE, ExcludeDummies = TRUE, Return = "All")
PairsG$Counts

# pairwise correct assignment rate:
PairsG$Summary[,"OK"] / PairsG$Summary[,"n"]

# check specific pair:
PairsG$Array[, "i190_2010_M", "i168_2009_F"]
# or
RelDF <- PairsG$Dataframe   # for brevity
RelDF[RelDF$id.A=="i190_2010_M" & RelDF$id.B=="i168_2009_F", ]

# Colony-style lists of full sib dyads & half sib dyads:
FullSibDyads <- with(RelDF, RelDF[Ped1 == "FS" & id.A < id.B, ])
HalfSibDyads <- with(RelDF, RelDF[Ped1 == "HS" & id.A < id.B, ])
# Use 'id.A < id.B' because each pair is listed 2x
}

}
\seealso{
\code{\link{PedCompare}} for individual-based comparison;
  \code{\link{GetRelM}} for a pairwise relationships matrix of a single
  pedigree; \code{\link{PlotRelPairs}} for visualisation of relationships
  within each pedigree.

  To estimate P(actual relationship (Ped1) | inferred relationship (Ped2)),
  see examples at \code{\link{EstConf}}.
}
