% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shadeHeight.R
\name{shadeHeight}
\alias{shadeHeight}
\title{Shade height calculation considering sun position and buildings outlines.}
\usage{
shadeHeight(location, build, height_field, solar_pos, b = 0.1,
  messages = TRUE)
}
\arguments{
\item{location}{A \code{SpatialPoints*} object specifying the location for which to calculate shade height}

\item{build}{A \code{SpatialPolygonsDataFrame} object specifying the buildings outline.}

\item{height_field}{The name of the column with building height in \code{build}}

\item{solar_pos}{A matrix with the solar azimuth (in degrees from North), and elevation}

\item{b}{Buffer size when joining intersection points with building outlines, to determine intersection height.}

\item{messages}{Whether a message regarding distance units of the CRS should be displayed.}
}
\value{
Shade height, in meters.
}
\description{
This function calculates shade height at a given point (\code{location}),
taking into account:\itemize{
\item{Buildings outline, given by a polygonal layer including a height attribute}
\item{Sun position, given by elevation and azimuth angles}
}
}
\note{
For a correct geometric calculation, make sure that:\itemize{
\item{The layers \code{location} and \code{build} are projected}
\item{The values in \code{height_field} of \code{build} are given in the same distance units as the CRS (e.g. meters when using UTM)}
}
}
\examples{
# Single location
location = rgeos::gCentroid(build)
time = as.POSIXct("2004-12-24 13:30:00", tz = "Asia/Jerusalem")
solar_pos = maptools::solarpos(
  matrix(c(34.7767978098526, 31.9665936050395), ncol = 2),
  time
  )
plot(build, main = time)
plot(location, add = TRUE)
sun = shadow:::.sunLocation(location = location, sun_az = solar_pos[1,1], sun_elev = solar_pos[1,2])
sun_ray = ray(from = location, to = sun)
build_outline = as(build, "SpatialLinesDataFrame")
inter = rgeos::gIntersection(build_outline, sun_ray)
plot(sun_ray, add = TRUE, col = "yellow")
plot(inter, add = TRUE, col = "red")
shadeHeight(location, build, "BLDG_HT", solar_pos)

\dontrun{

# Grid
ext = as(raster::extent(build), "SpatialPolygons")
r = raster::raster(ext, res = 3)
proj4string(r) = proj4string(build)
grid = raster::rasterToPoints(r, spatial = TRUE)
grid = sp::SpatialPointsDataFrame(grid, data.frame(grid_id = 1:length(grid)))
height_field = "BLDG_HT"
for(i in 1:length(grid)) {
  grid$shade_height[i] =
    shadeHeight(grid[i, ], build, height_field, solar_pos, messages = FALSE)
}
shade = as(grid, "SpatialPixelsDataFrame")
shade = raster::raster(shade, layer = "shade_height")
plot(shade, col = grey(seq(0.9, 0.2, -0.01)), main = time)
raster::contour(shade, add = TRUE)
plot(build, add = TRUE, border = "red")

}

}

