% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dimensionality_reduction.R
\name{PLS}
\alias{PLS}
\title{Partial Least Squares 'a la carte'}
\usage{
PLS(
  xdata,
  ydata,
  selectedX = NULL,
  selectedY = NULL,
  family = "gaussian",
  ncomp = NULL,
  scale = TRUE
)
}
\arguments{
\item{xdata}{matrix of predictors with observations as rows and variables as
columns.}

\item{ydata}{optional vector or matrix of outcome(s). If \code{family} is set
to \code{"binomial"} or \code{"multinomial"}, \code{ydata} can be a vector
with character/numeric values or a factor.}

\item{selectedX}{binary matrix of size \code{(ncol(xdata) * ncomp)}. The
binary entries indicate which predictors (in rows) contribute to the
definition of each component (in columns). If \code{selectedX=NULL}, all
predictors are selected for all components.}

\item{selectedY}{binary matrix of size \code{(ncol(ydata) * ncomp)}. The
binary entries indicate which outcomes (in rows) contribute to the
definition of each component (in columns). If \code{selectedY=NULL}, all
outcomes are selected for all components.}

\item{family}{type of PLS model. Only \code{family="gaussian"} is supported.
This corresponds to a PLS model as defined in \code{\link[mixOmics]{pls}}
(for continuous outcomes).}

\item{ncomp}{number of components.}

\item{scale}{logical indicating if the data should be scaled (i.e.
transformed so that all variables have a standard deviation of one).}
}
\value{
A list with: \item{Wmat}{matrix of X-weights.} \item{Wstar}{matrix of
  transformed X-weights.} \item{Pmat}{matrix of X-loadings.}
  \item{Cmat}{matrix of Y-weights.} \item{Tmat}{matrix of X-scores.}
  \item{Umat}{matrix of Y-scores.} \item{Qmat}{matrix needed for
  predictions.} \item{Rmat}{matrix needed for predictions.}
  \item{meansX}{vector used for centering of predictors, needed for
  predictions.} \item{sigmaX}{vector used for scaling of predictors, needed
  for predictions.} \item{meansY}{vector used for centering of outcomes,
  needed for predictions.} \item{sigmaY}{vector used for scaling of outcomes,
  needed for predictions.} \item{methods}{a list with \code{family} and
  \code{scale} values used for the run.} \item{params}{a list with
  \code{selectedX} and \code{selectedY} values used for the run.}
}
\description{
Runs a Partial Least Squares (PLS) model in regression mode using algorithm
implemented in \code{\link[mixOmics]{pls}}. This function allows for the
construction of components based on different sets of predictor and/or
outcome variables. This function is not using stability.
}
\details{
All matrices are defined as in (Wold et al. 2001). The weight matrix
  \code{Wmat} is the equivalent of \code{loadings$X} in
  \code{\link[mixOmics]{pls}}. The loadings matrix \code{Pmat} is the
  equivalent of \code{mat.c} in \code{\link[mixOmics]{pls}}. The score
  matrices \code{Tmat} and \code{Qmat} are the equivalent of
  \code{variates$X} and \code{variates$Y} in \code{\link[mixOmics]{pls}}.
}
\examples{
\donttest{
oldpar <- par(no.readonly = TRUE)

# Data simulation
set.seed(1)
simul <- SimulateRegression(n = 200, pk = c(5, 5, 5), family = "gaussian")
x <- simul$xdata
y <- simul$ydata

# PLS
mypls <- PLS(xdata = x, ydata = y, ncomp = 3)

# Sparse PLS to identify relevant variables
stab <- BiSelection(
  xdata = x, ydata = y,
  family = "gaussian", ncomp = 3,
  LambdaX = 1:(ncol(x) - 1),
  LambdaY = 1:(ncol(y) - 1),
  implementation = SparsePLS,
  n_cat = 2
)
plot(stab)

# Refitting of PLS model
mypls <- PLS(
  xdata = x, ydata = y,
  selectedX = stab$selectedX,
  selectedY = stab$selectedY
)

# Nonzero entries in weights are the same as in selectedX
par(mfrow = c(2, 2))
Heatmap(stab$selectedX,
  legend = FALSE
)
title("Selected in X")
Heatmap(ifelse(mypls$Wmat != 0, yes = 1, no = 0),
  legend = FALSE
)
title("Nonzero entries in Wmat")
Heatmap(stab$selectedY,
  legend = FALSE
)
title("Selected in Y")
Heatmap(ifelse(mypls$Cmat != 0, yes = 1, no = 0),
  legend = FALSE
)
title("Nonzero entries in Cmat")

# Multilevel PLS
if (requireNamespace("mixOmics", quietly = TRUE)) {
  # Generating random design
  z <- rep(1:50, each = 4)

  # Extracting the within-variability
  x_within <- mixOmics::withinVariation(X = x, design = cbind(z))

  # Running PLS on within-variability
  mypls <- PLS(xdata = x_within, ydata = y, ncomp = 3)
}

par(oldpar)
}
}
\references{
\insertRef{PLS}{sharp}
}
\seealso{
\code{\link{VariableSelection}}, \code{\link{BiSelection}}
}
