% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geno_last_gen.R
\name{geno_last_gen}
\alias{geno_last_gen}
\title{Draw random genotypes for last generation of a pedigree with known founder genotypes}
\usage{
geno_last_gen(X, fam, ids, missing_vals = c("", 0))
}
\arguments{
\item{X}{The genotype matrix of the founders (loci along rows, individuals along columns).
This matrix must have column names that identify each founder (matching codes in \code{fam$id}).
Individuals may be in a different order than \code{fam$id}.
Extra individuals in \code{admix} but absent in \code{fam$id} will be silently ignored.
All values should be in \code{c(0L, 1L, 2L)}; for speed, this code does not check that \code{X} is valid (i.e. fractional values between 0 and 2 may not cause errors).}

\item{fam}{The pedigree data.frame, in plink FAM format.
Only columns \code{id}, \code{pat}, and \code{mat} are required.
\code{id} must be unique and non-missing.
Founders must be present, and their \code{pat} and \code{mat} values must be missing (see below).
Non-founders must have both their parents be non-missing.
Parents must appear earlier than their children in the table.}

\item{ids}{A list containing vectors of IDs for each generation.
All these IDs must be present in \code{fam$id}.
If IDs in \code{fam} and \code{ids} do not fully agree, the code processes the IDs in the intersection, which is helpful when \code{fam} is pruned but \code{ids} is the original (larger) set.}

\item{missing_vals}{The list of ID values treated as missing.
\code{NA} is always treated as missing.
By default, the empty string ('') and zero (0) are also treated as missing (remove values from here if this is a problem).}
}
\value{
The random genotype matrix of the last generation (the intersection of \code{ids[ length(ids) ]} and \code{fam$id}).
The columns of this matrix are last-generation individuals in the order that they appear in \code{fam$id}.
The rows (loci) are the same as in the input \code{X}.
}
\description{
A wrapper around the more general \code{\link[=geno_fam]{geno_fam()}}, specialized to save memory when only the last generation is desired (\code{\link[=geno_fam]{geno_fam()}} returns genotypes for the entire pedigree in a single matrix).
This function assumes that generations are non-overlapping (met by the output of \code{\link[=sim_pedigree]{sim_pedigree()}}), in which case each generation \code{g} can be drawn from generation \code{g-1} data only.
That way, only two consecutive generations need be in memory at any given time.
The partitioning of individuals into generations is given by the \code{ids} parameter (again matches the output of \code{\link[=sim_pedigree]{sim_pedigree()}}).
}
\examples{
# A small pedigree, two parents and two children.
# A minimal fam table with the three required columns.
# Note "mother" and "father" have missing parent IDs, while children do not
library(tibble)
fam <- tibble(
  id = c('father', 'mother', 'child', 'sib'),
  pat = c(NA, NA, 'father', 'father'),
  mat = c(NA, NA, 'mother', 'mother')
)
# need an `ids` list separating the generations
ids <- list( c('father', 'mother'), c('child', 'sib') )

# genotypes of the parents at 4 loci
X <- cbind( c(1, 2, 0, 2), c(0, 2, 2, 1) )
# Name the parents with same codes as in `fam`
# (order can be different)
colnames( X ) <- c('mother', 'father')
# name loci too
rownames( X ) <- paste0( 'rs', 1:4 )

# Draw the genotype matrix of the children
X2 <- geno_last_gen( X, fam, ids )
X2

}
\seealso{
Plink FAM format reference:
\url{https://www.cog-genomics.org/plink/1.9/formats#fam}
}
