% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fh_weight.R
\name{fh_weight}
\alias{fh_weight}
\title{Fleming-Harrington weighted logrank tests}
\usage{
fh_weight(
  x = counting_process(cut_data_by_event(sim_pw_surv(n = 200), 150), arm =
    "experimental"),
  rho_gamma = data.frame(rho = c(0, 0, 1, 1), gamma = c(0, 1, 0, 1)),
  return_variance = FALSE,
  return_corr = FALSE
)
}
\arguments{
\item{x}{A \code{\link[=counting_process]{counting_process()}}-class data frame with a counting process
dataset.}

\item{rho_gamma}{A data frame with variables \code{rho} and \code{gamma}, both greater
than equal to zero, to specify one Fleming-Harrington weighted logrank test
per row; Default: \code{data.frame(rho = c(0, 0, 1, 1), gamma = c(0, 1, 0, 1))}.}

\item{return_variance}{A logical flag that, if \code{TRUE}, adds columns
estimated variance for weighted sum of observed minus expected;
see details; Default: \code{FALSE}.}

\item{return_corr}{A logical flag that, if \code{TRUE}, adds columns
estimated correlation for weighted sum of observed minus expected;
see details; Default: \code{FALSE}.}
}
\value{
A data frame with \code{rho_gamma} as input and the FH test statistic
for the data in \code{x}. (\code{z}, a directional square root of the usual
weighted logrank test); if variance calculations are specified
(for example, to be used for covariances in a combination test),
then this will be returned in the column \code{Var}.
}
\description{
With output from the function \code{\link[=counting_process]{counting_process()}}.
}
\details{
The input value \code{x} produced by \code{\link[=counting_process]{counting_process()}} produces a
counting process dataset grouped by stratum and sorted within stratum
by increasing times where events occur.
\itemize{
\item \eqn{z} - Standardized normal Fleming-Harrington weighted logrank test.
\item \eqn{i} - Stratum index.
\item \eqn{d_i} - Number of distinct times at which events occurred in
stratum \eqn{i}.
\item \eqn{t_{ij}} - Ordered times at which events in stratum
\eqn{i}, \eqn{j = 1, 2, \ldots, d_i} were observed;
for each observation, \eqn{t_{ij}} represents the time post study entry.
\item \eqn{O_{ij.}} - Total number of events in stratum \eqn{i} that occurred
at time \eqn{t_{ij}}.
\item \eqn{O_{ije}} - Total number of events in stratum \eqn{i} in the
experimental treatment group that occurred at time \eqn{t_{ij}}.
\item \eqn{N_{ij.}} - Total number of study subjects in stratum \eqn{i}
who were followed for at least duration.
\item \eqn{E_{ije}} - Expected observations in experimental treatment group
given random selection of \eqn{O_{ij.}} from those in
stratum \eqn{i} at risk at time \eqn{t_{ij}}.
\item \eqn{V_{ije}} - Hypergeometric variance for \eqn{E_{ije}} as
produced in \code{Var} from \code{\link[=counting_process]{counting_process()}}.
\item \eqn{N_{ije}} - Total number of study subjects in
stratum \eqn{i} in the experimental treatment group
who were followed for at least duration \eqn{t_{ij}}.
\item \eqn{E_{ije}} - Expected observations in experimental group in
stratum \eqn{i} at time \eqn{t_{ij}} conditioning on the overall number
of events and at risk populations at that time and sampling at risk
observations without replacement:
\deqn{E_{ije} = O_{ij.} N_{ije}/N_{ij.}}
\item \eqn{S_{ij}} - Kaplan-Meier estimate of survival in combined
treatment groups immediately prior to time \eqn{t_{ij}}.
\item \eqn{\rho, \gamma} - Real parameters for Fleming-Harrington test.
\item \eqn{X_i} - Numerator for signed logrank test in stratum \eqn{i}
\deqn{X_i = \sum_{j=1}^{d_{i}} S_{ij}^\rho(1-S_{ij}^\gamma)(O_{ije}-E_{ije})}
\item \eqn{V_{ij}} - Variance used in denominator for Fleming-Harrington
weighted logrank tests
\deqn{V_i = \sum_{j=1}^{d_{i}} (S_{ij}^\rho(1-S_{ij}^\gamma))^2V_{ij})}
The stratified Fleming-Harrington weighted logrank test is then computed as:
\deqn{z = \sum_i X_i/\sqrt{\sum_i V_i}.}
}
}
\examples{
library(dplyr)

# Example 1
# Use default enrollment and event rates at cut at 100 events
x <- sim_pw_surv(n = 200) |>
  cut_data_by_event(100) |>
  counting_process(arm = "experimental")

# Compute logrank FH(0, 1)
fh_weight(x, rho_gamma = data.frame(rho = 0, gamma = 1))
fh_weight(x, rho_gamma = data.frame(rho = 0, gamma = 1), return_variance = TRUE)

# Compute the corvariance between FH(0, 0), FH(0, 1) and FH(1, 0)
fh_weight(x, rho_gamma = data.frame(rho = c(0, 0, 1), gamma = c(0, 1, 0)))
fh_weight(x, rho_gamma = data.frame(rho = c(0, 0, 1), gamma = c(0, 1, 0)), return_variance = TRUE)
fh_weight(x, rho_gamma = data.frame(rho = c(0, 0, 1), gamma = c(0, 1, 0)), return_corr = TRUE)

# Example 2
# Use default enrollment and event rates at cut of 100 events
set.seed(123)
x <- sim_pw_surv(n = 200) |>
  cut_data_by_event(100) |>
  counting_process(arm = "experimental") |>
  fh_weight(rho_gamma = data.frame(rho = c(0, 0), gamma = c(0, 1)), return_corr = TRUE)

# Compute p-value for MaxCombo
library(mvtnorm)
1 - pmvnorm(
  lower = rep(min(x$z), nrow(x)),
  corr = data.matrix(select(x, -c(rho, gamma, z))),
  algorithm = GenzBretz(maxpts = 50000, abseps = 0.00001)
)[1]

# Check that covariance is as expected
x <- sim_pw_surv(n = 200) |>
  cut_data_by_event(100) |>
  counting_process(arm = "experimental")

x |> fh_weight(
  rho_gamma = data.frame(
    rho = c(0, 0),
    gamma = c(0, 1)
  ),
  return_variance = TRUE
)

# Off-diagonal element should be variance in following
x |> fh_weight(
  rho_gamma = data.frame(
    rho = 0,
    gamma = .5
  ),
  return_variance = TRUE
)

# Compare off diagonal result with fh_weight()
x |> fh_weight(rho_gamma = data.frame(rho = 0, gamma = .5))
}
