#' Display Avatar
#'
#' Given an avatar ID, grab the picture (full or thumbnail) concerning that particular avatar.
#' Note that this function downloads the image into your temporary directory to display 
#' it. Then, the function deletes the file before completing. If it would be preferred to keep
#' the image at the temporary location, set the keep_image argument to TRUE (default is FALSE).
#'
#' @return Returns a picture displaying the avatar picture
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, October 2021
#' @keywords avatar picture
#' @importFrom utils download.file
#' @export
#' @examples
#' \dontrun{get_avatar_picture("c751b27d9158c1dd41bd33dc7e4bcdde",
#'    type = "full",
#'    keep_image = FALSE)}
#'
#' @param avatar_id Avatar ID generated by Sleeper (character)
#' @param type String that is either "full" or "thumbnail"
#' @param keep_image FALSE to delete the image after displaying it or TRUE to keep it
#'
get_avatar_picture <- function(avatar_id, type = "full", keep_image = FALSE) {
  # Generate a name for the temporary file
  temp_fil <- paste0(tempfile(), ".png")
  # If the type is "full" then get full PNG file
  if(type == "full") {
    # Get picture from API
    utils::download.file(paste0("https://sleepercdn.com/avatars/", avatar_id), destfile = temp_fil, quiet = TRUE)
    # Show file
    file.show(temp_fil)
    # Check whether image should deleted
    if(!keep_image) {
      # If should be deleted, remove the file
      invisible(file.remove(temp_fil))
      # Inform user the file was removed
      message("File successfully removed from temporary location.")
    } else {
      # If should be kept, inform user the file was kept and where to find it
      message(paste0("Temporary file not deleted. It can be found at ", temp_fil, "."))
    }
  } # If the type is "thumbnail" then get full PNG file
  else if(type == "thumbnail") {
    # Get picture from API
    download.file(paste0("https://sleepercdn.com/avatars/thumbs/", avatar_id), destfile = temp_fil, quiet = TRUE)
    # Show file
    file.show(temp_fil)
    # Check whether image should deleted
    if(!keep_image) {
      # If should be deleted, remove the file
      invisible(file.remove(temp_fil))
      # Inform user the file was removed
      message("File successfully removed from temporary location.")
    } else {
      # If should be kept, inform user the file was kept and where to find it
      message(paste0("Temporary file not deleted. It can be found at ", temp_fil, "."))
    }
  }
  else {
    # If type specified is not valid, throw error message to user
    stop("Invalid value entered for type: can only be 'full' or 'thumbnail'")
  }
}
