% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msprime.R
\name{msprime}
\alias{msprime}
\title{Run a slendr model in msprime}
\usage{
msprime(
  model,
  sequence_length,
  recombination_rate,
  samples = NULL,
  random_seed = NULL,
  verbose = FALSE,
  debug = FALSE,
  run = TRUE,
  path = NULL
)
}
\arguments{
\item{model}{Model object created by the \code{compile} function}

\item{sequence_length}{Total length of the simulated sequence (in base-pairs)}

\item{recombination_rate}{Recombination rate of the simulated sequence (in
recombinations per basepair per generation)}

\item{samples}{A data frame of times at which a given number of individuals
should be remembered in the tree-sequence (see \code{schedule_sampling} for a
function that can generate the sampling schedule in the correct format). If
missing, only individuals present at the end of the simulation will be
recorded in the final tree-sequence file.}

\item{random_seed}{Random seed (if \code{NULL}, a seed will be generated between
0 and the maximum integer number available)}

\item{verbose}{Write the log information from the SLiM run to the console
(default \code{FALSE})?}

\item{debug}{Write msprime's debug log to the console (default \code{FALSE})?}

\item{run}{Should the msprime engine be run? If \code{FALSE}, the command line msprime
command will be printed (and returned invisibly as a character vector) but not executed.}

\item{path}{Path to the directory where simulation result files will be saved.
If \code{NULL}, this directory will be automatically created as a temporary
directory. If \code{TRUE}, this path will be also returned by the function.
If a string is given, it is assumed to be a path to a directory where simulation
results will be saved. In this case, the function will return this path invisibly.
Note that if a tree-sequence file should be simulated (along with other files,
potentially), that tree-sequence file (named 'msprime.trees' by default) will
have to be explicitly loaded using \code{ts_read()}.}
}
\value{
A tree-sequence object loaded via Python-R reticulate interface function \code{ts_read}
(internally represented by the Python object \code{tskit.trees.TreeSequence}). If the
\code{path} argument was set, it will return the path as a single-element character vector.
}
\description{
This function will execute a built-in msprime script and run a compiled
slendr demographic model.
}
\examples{
\dontshow{check_dependencies(python = TRUE, quit = TRUE) # dependencies must be present
}
init_env()

# load an example model
model <- read_model(path = system.file("extdata/models/introgression", package = "slendr"))

# afr and eur objects would normally be created before slendr model compilation,
# but here we take them out of the model object already compiled for this
# example (in a standard slendr simulation pipeline, this wouldn't be necessary)
afr <- model$populations[["AFR"]]
eur <- model$populations[["EUR"]]
chimp <- model$populations[["CH"]]

# schedule the sampling of a couple of ancient and present-day individuals
# given model at 20 ky, 10 ky, 5ky ago and at present-day (time 0)
modern_samples <- schedule_sampling(model, times = 0, list(afr, 10), list(eur, 100), list(chimp, 1))
ancient_samples <- schedule_sampling(model, times = c(40000, 30000, 20000, 10000), list(eur, 1))

# sampling schedules are just data frames and can be merged easily
samples <- rbind(modern_samples, ancient_samples)

# run a simulation using the msprime back end from a compiled slendr model object
ts <- msprime(model, sequence_length = 1e5, recombination_rate = 0, samples = samples)

# simulated tree-sequence object can be saved to a file using ts_write()...
ts_file <- normalizePath(tempfile(fileext = ".trees"), winslash = "/", mustWork = FALSE)
ts_write(ts, ts_file)
# ... and, at a later point, loaded by ts_read()
ts <- ts_read(ts_file, model)

summary(ts)
}
