% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{glog}
\alias{glog}
\title{Generalized linear model constraint on hierarchical structure
by using overlapped group penalty}
\usage{
glog(y, x, g, v, lambda, hierarchy, family = "gaussian", rho = 10,
  scale = TRUE, eabs = 0.001, erel = 0.001, LL = 1, eta = 1.25,
  maxitr = 1000L)
}
\arguments{
\item{y}{response variable, in the format of matrix. When family is
\code{gaussian'' or }binomial'', \code{y} ought to
be a matrix of n by 1 for the observations; when family
is ``coxph'', y represents the survival objects, that is,
a matrix of n by 2, containing the survival time and the censoring status.
See \code{\link[survival]{Surv}}.}

\item{x}{a model matrix of dimensions n by p,in which the column represents
the predictor variables.}

\item{g}{a numeric vector of group labels for the predictor variables.}

\item{v}{a numeric vector of binary values, represents whether or not the
predictor variables are penalized. Note that 1 indicates
penalization and 0 for not penalization.}

\item{lambda}{a numeric vector of three penalty parameters corresponding to L2 norm,
squared L2 norm, and L1 norm, respectively.}

\item{hierarchy}{a factor value in levels 0, 1, 2, which represent different
hierarchical structure within groups, respectively.
When \code{hierarchy=0}, \eqn{\lambda_2} and \eqn{\lambda_3} are
forced to be zeroes; when \code{hierarchy=1}, \eqn{\lambda_2} is
forced to be zero; when \code{hierarchy=2}, there is no constraint
on \eqn{\lambda}'s. See \code{\link{smog}}.}

\item{family}{a description of the distribution family for the response
variable variable. For continuous response variable,
family is \code{gaussian''; for multinomial or binary response variable, family is }binomial''; for survival response
variable, family is ``coxph'', respectively.}

\item{rho}{the penalty parameter used in the alternating direction method
of multipliers algorithm (ADMM). Default is 10.}

\item{scale}{whether or not scale the design matrix. Default is \code{true}.}

\item{eabs}{the absolute tolerance used in the ADMM algorithm. Default is 1e-3.}

\item{erel}{the reletive tolerance used in the ADMM algorithm. Default is 1e-3.}

\item{LL}{initial value for the Lipschitz continuous constant for
approximation to the objective function in the Majorization-
Minimization (MM) (or iterative shrinkage-thresholding algorithm
(ISTA)). Default is 1.}

\item{eta}{gradient stepsize for the backtrack line search for the Lipschitz
continuous constant. Default is 1.25.}

\item{maxitr}{the maximum iterations for convergence in the ADMM algorithm.
Default is 500.}
}
\value{
A list of
\item{coefficients}{A data frame of the variable name and the estimated coefficients}
\item{llikelihood}{The log likelihood value based on the ultimate estimated coefficients}
\item{loglike}{The sequence of log likelihood values since the algorithm starts}
\item{PrimalError}{The sequence of primal errors in the ADMM algorithm}
\item{DualError}{The sequence of dual errors in the ADMM algorithm}
\item{converge}{The integer of the iteration when the convergence occurs}
}
\description{
Generalized linear model constraint on hierarchical structure
by using overlapped group penalty
}
\examples{

set.seed(2018) 
# generate design matrix x
n=50;p=100
s=10
x=matrix(0,n,1+2*p)
x[,1]=sample(c(0,1),n,replace = TRUE)
x[,seq(2,1+2*p,2)]=matrix(rnorm(n*p),n,p)
x[,seq(3,1+2*p,2)]=x[,seq(2,1+2*p,2)]*x[,1]

g=c(p+1,rep(1:p,rep(2,p)))  # groups 
v=c(0,rep(1,2*p))           # penalization status

# generate beta
beta=c(rnorm(13,0,2),rep(0,ncol(x)-13))
beta[c(2,4,7,9)]=0

# generate y
data1=x\%*\%beta
noise1=rnorm(n)
snr1=as.numeric(sqrt(var(data1)/(s*var(noise1))))
y1=data1+snr1*noise1
lambda = c(8,0,8)
hierarchy = 1
gfit1 = glog(y1,x,g,v,lambda,hierarchy,family = "gaussian")

}
\references{
\insertRef{ma2019structural}{smog}
}
\seealso{
\code{\link{cv.smog}}, \code{\link{smog.default}}, \code{\link{smog.formula}},
\code{\link{predict.smog}}, \code{\link{plot.smog}}.
}
\author{
Chong Ma, \email{chongma8903@gmail.com}.
}
