% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analysis.R
\name{get_stats}
\alias{get_stats}
\title{Get estimates of time-dependent properties of models.}
\usage{
get_stats(
  x,
  center_trend = "grand_mean",
  center_kernel = c("gaussian", "uniform"),
  center_bandwidth = NULL,
  stat_trend = c("local_constant", "local_linear"),
  stat_kernel = c("uniform", "gaussian"),
  stat_bandwidth = NULL,
  lag = 1,
  backward_only = FALSE
)
}
\arguments{
\item{x}{A univariate or multivariate numeric time series object or
a numeric vector or matrix.}

\item{center_trend}{Character string giving method of calculating
the trend to subtract. Allowed values are '"assume_zero"',
'"grand_mean"', '"ensemble_means"', '"local_constant"', and
'"local_linear"'. Will be partially matched.}

\item{center_kernel}{Character string giving the kernel for any
local detrending. Allowed values are '"gaussian"' and
'"uniform"'.}

\item{center_bandwidth}{Bandwidth of kernel for any local detrending
done. A numeric value >= 1.}

\item{stat_trend}{Character string giving method of smoothing
estimates. Allowed values are '"local_constant"', and
'"local_linear"'. Will be partially matched.}

\item{stat_kernel}{Character string giving the kernel for local
smoothing of estimates. Allowed values are '"gaussian"' and
'"uniform"'.}

\item{stat_bandwidth}{Bandwidth of kernel for local smoothing of
estimates.  A numeric value >= 1.}

\item{lag}{Integer lag at which to calculate the acf. This lag is
in terms of the index of \code{x} and does not account for the
frequency of \code{x} if \code{x} is a time series. It should
be non-negative.}

\item{backward_only}{Logical value (defaulting to 'FALSE') that
determines whether any uniform smoothing kernels are restricted
to using data before the index of the smoothed estimate.}
}
\value{
A list with elements '"stats"', '"taus"', '"centered"',
    '"stat_trend"', '"stat_kernel"', '"stat_bandwidth"', and
    '"lag"'. "stats" is a list containing vectors of the
    estimates. '"taus"' is a list containing Kendall's correlation
    coefficient of each element of '"stats"' with
    time. '"centered"' is a list of the detrended time series, the
    trend subtracted, and the bandwidth used in the detrending. The
    other elements record the parameters provided to this function
    for future reference.
}
\description{
\code{get_stats} estimates time-dependent properties of models
(e.g., variance) from ensemble time series.
}
\details{
Any missing values in 'x' will cause an error.

Bandwidths affect weights in local smoothers as follows. To get the
local estimate corresponding to index i, the distance to each other
index j is calculated as (i - j) / h, where h is the
bandwidth. Then that distance is plugged into the kernel function
to obtain a weight. The weights are normalized to sum to one for
each index.

The gaussian kernel is equivalent to a standard Gaussian density
function. The uniform kernel is an indicator function of whether
the distance is less than 1. Thus selecting a uniform kernel with a
bandwidth of 2 is equivalent to a sliding window of length 3 that
is centered on the focal index. In general, if n is the greatest
integer that is less than the value of the bandwidth h, the window
includes the n nearest values on each side of the focal index.

'"local_constant"' smoothers are local means computed with the
kernel weights. '"local_linear"' smoothers are the fitted values of
local linear regressions with the kernel weights. The linear
smoothers avoid biases that the one-sided kernels at the ends of
the time series can create for the local constant smoothers.

See the vignette "Getting Started with spaero" for the formulas
used for each estimate.
}
\examples{

# A highly autocorrelated time series
x <- 1:10
get_stats(x, stat_bandwidth = 3)$stats

# Plot log of acf
plot(log(get_stats(x, stat_bandwidth = 3)$stats$autocor))

# Check estimates with AR1 simulations with lag-1 core 0.1
w <- rnorm(1000)
xnext <- function(xlast, w) 0.1 * xlast + w
x <- Reduce(xnext, x = w, init = 0, accumulate = TRUE)
acf(x, lag.max = 1, plot = FALSE)
head(get_stats(x, stat_bandwidth = length(x))$stats$autocor)

# Check detrending ability
x2 <- x + seq(1, 10, len = length(x))
ans <- get_stats(x2, center_trend = "local_linear",
                  center_bandwidth = length(x),
                   stat_bandwidth = length(x))$stats
head(ans$autocor)

# The simple acf estimate is inflated by the trend
acf(x2, lag.max = 1, plot = FALSE)

# Check ability to estimate time-dependent autocorrelation
xnext <- function(xlast, w) 0.8 * xlast + w
xhi <- Reduce(xnext, x = w, init = 0, accumulate = TRUE)
acf(xhi, lag.max = 1, plot = FALSE)
wt <- seq(0, 1, len = length(x))
xdynamic <- wt * xhi + (1 - wt) * x
get_stats(xdynamic, stat_bandwidth = 100)$stats$autocor
}
\seealso{
\code{\link{acf}}, \code{\link{var}},
\code{\link[moments]{kurtosis}}, and
\code{\link[moments]{skewness}} for estimation of properties that
are not time-dependent. See
\code{\link[earlywarnings]{generic_ews}} for another approach to
estimation of time-dependent properties.
}
