## ------------------------------------------------------------------------------------------------|
#     Copyright (C) 2018  Reza Mohammadi                                                           |
#                                                                                                  |
#     This file is part of ssgraph package.                                                        |
#                                                                                                  |
#     ssgraph is free software: you can redistribute it and/or modify it under                     |
#     the terms of the GNU General Public License as published by the Free                         |
#     Software Foundation; see <https://cran.r-project.org/web/licenses/GPL-3>.                    |
#                                                                                                  |
#     Maintainer: Reza Mohammadi <a.mohammadi@uva.nl>                                              |
## ------------------------------------------------------------------------------------------------|
#  R code for Graphcial models based on spike and slab priors                                      |
## ------------------------------------------------------------------------------------------------|

ssgraph = function( data, n = NULL, method = "ggm", iter = 5000, burnin = iter / 2, 
                    var1 = 4e-04, var2 = 1, lambda = 1, g.prior = 0.5, 
                    g.start = "full", sig.start = NULL, save.all = FALSE, print = 1000, 
                    cores = 2 )
{
    if( iter < burnin ) stop( "Number of iteration must be more than number of burn-in" )
    if( ( g.prior <= 0 ) | ( g.prior >= 1 ) ) stop( "'g.prior' must be between 0 and 1" )   
    if(  var1 <= 0 ) stop( "'var1' must be more than 0" )
    if(  var2 <= 0 ) stop( "'var2' must be more than 0" )

    check.os( os = 2 )	
    if( cores == "all" ) cores = BDgraph::detect_cores()
    
    tmp   <- .C( "check_nthread", cores = as.integer(cores), PACKAGE = "ssgraph" )
    cores <- tmp $ cores
    
    .C( "omp_set_num_cores", as.integer( cores ), PACKAGE = "ssgraph" )
    
    burnin <- floor( burnin )
    
    if( class( data ) == "sim" ) data <- data $ data
    
    if( !is.matrix( data ) & !is.data.frame( data ) ) stop( "Data must be a matrix or dataframe" )
    if( is.data.frame( data ) ) data <- data.matrix( data )
    
    if( any( is.na( data ) ) ) 
    {
        if( method == "ggm" ) { stop( "ggm method does not deal with missing values. You could choose option method = gcgm" ) }	
        gcgm_NA = 1
    }else{
        gcgm_NA = 0
    }
    
    dimd <- dim( data )
    p    <- dimd[ 2 ]
    if( p < 3 ) stop( "Number of variables/nodes ('p') must be more than or equal with 2" )
    if( is.null( n ) ) n <- dimd[ 1 ]

    if( method == "gcgm" )
    {
        if( isSymmetric( data ) ) stop( "method='gcgm' requires all data" )
        
        R <- 0 * data
        for( j in 1:p ) R[ , j ] = match( data[ , j ], sort( unique( data[ , j ] ) ) ) 
        R[ is.na( R ) ] = 0     # dealing with missing values	
        
        # copula for continuous non-Gaussian data
        if( gcgm_NA == 0 && min( apply( R, 2, max ) ) > ( n - 5 * n / 100 ) )
        {
            # copula transfer 
            data = qnorm( apply( data, 2, rank ) / ( n + 1 ) )
            
            method = "ggm"
        }else{	
            # for non-Gaussian data
            Z                  <- qnorm( apply( data, 2, rank, ties.method = "random" ) / ( n + 1 ) )
            Zfill              <- matrix( rnorm( n * p ), n, p )       # for missing values
            Z[ is.na( data ) ] <- Zfill[ is.na( data ) ]               # for missing values
            Z                  <- t( ( t( Z ) - apply( Z, 2, mean ) ) / apply( Z, 2, sd ) )
            S                  <- t( Z ) %*% Z
        }
    } 
    
    if( method == "ggm" ) 
    {
        if( isSymmetric( data ) )
        {
            if ( is.null( n ) ) stop( "Please specify the number of observations 'n'" )
            cat( "Input is identified as the covriance matrix. \n" )
            S <- data
        }else{
            S <- t( data ) %*% data
        }
    }
    
    if( class( g.start ) == "bdgraph" ) 
    {
        G <- g.start $ last_graph
        K <- g.start $ last_K
    } 

    if( class( g.start ) == "ssgraph" ) 
    {
        G <- g.start $ last_graph
        K <- g.start $ last_K
    } 

    if( class( g.start ) == "sim" ) 
    {
        G <- as.matrix( g.start $ G )
        K <- as.matrix( g.start $ K )
    } 
    
    if( class( g.start ) == "character" && g.start == "empty"  ) G = matrix( 0, p, p )
    if( class( g.start ) == "character" && g.start == "full"   ) G = matrix( 1, p, p )
    if( is.matrix( g.start ) ) G = g.start
    
    if( ( sum( G == 0 ) + sum( G == 1 ) ) != ( p ^ 2 ) ) stop( "Element of 'g.start', as a matrix, must have 0 or 1" )
    
    diag( G ) = 0    
    if( !isSymmetric( G ) )
    {
        G[ lower.tri( G, diag( TRUE ) ) ] <- 0
        G  = G + t( G )
    }
    
    if( is.null( sig.start ) ) sigma = S else sigma = sig.start
    K = solve( sigma )      # precision or concentration matrix (omega)
    
    if( save.all == TRUE )
    {
        qp1           = ( p * ( p - 1 ) / 2 ) + 1
        string_g      = paste( c( rep( 0, qp1 ) ), collapse = '' )
        sample_graphs = c( rep ( string_g, iter - burnin ) )  # vector of numbers like "10100" 
        graph_weights = c( rep ( 0, iter - burnin ) )         # waiting time for every state
        all_graphs    = c( rep ( 0, iter - burnin ) )         # vector of numbers like "10100"
        all_weights   = c( rep ( 1, iter - burnin ) )         # waiting time for every state		
        size_sample_g = 0
    }
    
    if( ( save.all == TRUE ) && ( p > 50 & iter > 20000 ) )
    {
        cat( "  WARNING: Memory needs to run this function is around " )
        print( ( iter - burnin ) * object.size( string_g ), units = "auto" ) 
    } 
    
    nmc     = iter - burnin
    p_links = matrix( 0, p, p )
    K_hat   = matrix( 0, p, p )
    
    g_prior     = g.prior
    one_g_prior = 1 - g_prior

    mes <- paste( c( iter, " iteration is started.                    " ), collapse = "" )
    cat( mes, "\r" )
    
## ---- main BDMCMC algorithms implemented in C++ -------------------------------------------------|
    if( save.all == FALSE )
    { 
        if( method == "ggm" )
        {
             result = .C( "ggm_spike_slab_ma", as.integer(iter), as.integer(burnin), G = as.integer(G), K = as.double(K), as.double(S), as.integer(p), 
                         K_hat = as.double(K_hat), p_links = as.double(p_links), as.integer(n),
                         as.double(var1), as.double(var2), as.double(lambda), as.double(g_prior), as.integer(print), PACKAGE = "ssgraph" )
        }
        
        if( method == "gcgm" )
        {
             result = .C( "gcgm_spike_slab_ma", as.integer(iter), as.integer(burnin), G = as.integer(G), K = as.double(K), as.double(S), as.integer(p), 
                         K_hat = as.double(K_hat), p_links = as.double(p_links), as.integer(n),
                         as.double(Z), as.integer(R), as.integer(gcgm_NA),
                         as.double(var1), as.double(var2), as.double(lambda), as.double(g_prior), as.integer(print), PACKAGE = "ssgraph" )
        }
    }else{
        if( method == "ggm" )
        {
            result = .C( "ggm_spike_slab_map", as.integer(iter), as.integer(burnin), G = as.integer(G), K = as.double(K), as.double(S), as.integer(p), 
                         K_hat = as.double(K_hat), p_links = as.double(p_links), as.integer(n),
                         all_graphs = as.integer(all_graphs), all_weights = as.double(all_weights), 
                         sample_graphs = as.character(sample_graphs), graph_weights = as.double(graph_weights), size_sample_g = as.integer(size_sample_g),
                         as.double(var1), as.double(var2), as.double(lambda), as.double(g_prior), as.integer(print), PACKAGE = "ssgraph" )
        }
        
        if( method == "gcgm" )
        {
            result = .C( "gcgm_spike_slab_map", as.integer(iter), as.integer(burnin), G = as.integer(G), K = as.double(K), as.double(S), as.integer(p), 
                         K_hat = as.double(K_hat), p_links = as.double(p_links), as.integer(n),
                         all_graphs = as.integer(all_graphs), all_weights = as.double(all_weights), 
                         sample_graphs = as.character(sample_graphs), graph_weights = as.double(graph_weights), size_sample_g = as.integer(size_sample_g),
                         as.double(Z), as.integer(R), as.integer(gcgm_NA),
                         as.double(var1), as.double(var2), as.double(lambda), as.double(g_prior), as.integer(print), PACKAGE = "ssgraph" )
        }
    }
## ------------------------------------------------------------------------------------------------|
    
    label      = colnames( data )
    p_links    = matrix( result $ p_links, p, p, dimnames = list( label, label ) ) 
    K_hat      = matrix( result $ K_hat  , p, p, dimnames = list( label, label ) ) 
    
    last_graph = matrix( result $ G      , p, p, dimnames = list( label, label ) )
    last_K     = matrix( result $ K      , p, p, dimnames = list( label, label ) )

    diag( p_links ) = nmc
    p_links[ lower.tri( p_links ) ] = 0
    p_links = p_links / nmc
    K_hat = K_hat / nmc

    if( save.all == TRUE )
    {
        size_sample_g = result $ size_sample_g
        sample_graphs = result $ sample_graphs[ 1 : size_sample_g ]
        graph_weights = result $ graph_weights[ 1 : size_sample_g ]
        all_graphs    = result $ all_graphs + 1
        all_weights   = result $ all_weights

        output = list( p_links = p_links, K_hat = K_hat, last_graph = last_graph, last_K = last_K,
                       sample_graphs = sample_graphs, graph_weights = graph_weights, 
                       all_graphs = all_graphs, all_weights = all_weights )
    }else{
        output = list( p_links = p_links, K_hat = K_hat, last_graph = last_graph, last_K = last_K )
    }
    
    class( output ) = "ssgraph"
    return( output )
}

## ------------------------------------------------------------------------------------------------|
#    Summary of ssgraph output
## ------------------------------------------------------------------------------------------------|
summary.ssgraph = function( object, round = 2, vis = TRUE, ... )
{
    p_links    = object $ p_links
    p          = nrow( object $ last_graph )
    label      = colnames( object $ last_graph )
    if ( is.null( label ) ) label <- as.character( 1 : p )
    selected_g = matrix( 0, p, p, dimnames = list( label, label ) )	
    
    selected_g[ p_links >  0.5 ] = 1
    selected_g[ p_links <= 0.5 ] = 0
    
    if( vis )
    {
        # plot selected graph (graph with the highest posterior probability)
        G  <- igraph::graph.adjacency( selected_g, mode = "undirected", diag = FALSE )
        
        if( !is.null( object $ graph_weights ) ) 
            op = par( mfrow = c( 2, 2 ), pty = "s", omi = c( 0.3, 0.3, 0.3, 0.3 ), mai = c( 0.3, 0.3, 0.3, 0.3 ) ) 

        subGraph = "Selected graph with edge posterior probability = 0.5"
        
        if( p < 20 ) size = 15 else size = 2
        igraph::plot.igraph( G, layout = igraph::layout.circle, main = "Selected graph", sub = subGraph, vertex.color = "white", vertex.size = size, vertex.label.color = 'black' )
        
        if( !is.null( object $ graph_weights ) )
        {
            sample_graphs = object $ sample_graphs
            graph_weights = object $ graph_weights
            max_gWeights  = max( graph_weights )
            sum_gWeights  = sum( graph_weights )
            max_prob_G    = max_gWeights / sum_gWeights
            
            # plot posterior distribution of graph
            plot( x = 1 : length( graph_weights ), y = graph_weights / sum_gWeights, type = "h", main = "Posterior probability of graphs",
                  ylab = "Pr(graph|data)", xlab = "graph" )
            
            abline( h = max_prob_G, col = "red" )
            text( which( max_gWeights == graph_weights )[1], max_prob_G, "Pr(selected graph|data)", col = "gray60", adj = c( 0, +1 ) )
            
            # plot posterior distribution of graph size
            sizesample_graphs = sapply( sample_graphs, function( x ) length( which( unlist( strsplit( as.character(x), "" ) ) == 1 ) ) )
            xx       <- unique( sizesample_graphs )
            weightsg <- vector()
            
            for( i in 1 : length(xx) ) weightsg[i] <- sum( graph_weights[ which( sizesample_graphs == xx[i] ) ] )
            
            plot( x = xx, y = weightsg / sum_gWeights, type = "h", main = "Posterior probability of graphs size", ylab = "Pr(graph size|data)", xlab = "Graph size" )
            
            # plot trace of graph size
            all_graphs     = object $ all_graphs
            sizeall_graphs = sizesample_graphs[ all_graphs ]
            
            plot( x = 1 : length( all_graphs ), sizeall_graphs, type = "l", main = "Trace of graph size", ylab = "Graph size", xlab = "Iteration" )
            
            abline( h = sum( selected_g ), col = "red" )	  
            
            par( op )
        }
    }
    
    K_hat = object $ K_hat
    
    return( list( selected_g = selected_g, p_links = round( p_links, round ), K_hat = round( K_hat, round ) ) )
}  

## ------------------------------------------------------------------------------------------------|
#    Plot for class ssgraph
## ------------------------------------------------------------------------------------------------|
plot.ssgraph = function( x, cut = NULL, number.g = 1, layout = layout.circle, ... )
{
    if( is.null( cut ) ) cut = 0.5

    if( ( cut < 0 ) || ( cut > 1 ) ) stop( "Value of 'cut' must be between 0 and 1." )
    
    p_links                      = x $ p_links
    selected_g                   = 0 * p_links
    selected_g[ p_links > cut ]  = 1
    selected_g[ p_links <= cut ] = 0		
    
    G = igraph::graph.adjacency( selected_g, mode = "undirected", diag = FALSE )
    igraph::plot.igraph( G, layout = layout, main = "Selected graph", sub = paste0( "Edge posterior probability = ", cut ), ... )	   		
}

## ------------------------------------------------------------------------------------------------|
#    Print of the ssgraph output
## ------------------------------------------------------------------------------------------------|
print.ssgraph = function( x, round = 2, ... )
{
    p_links = x $ p_links
    
    selected_g                   = 0 * p_links
    selected_g[ p_links > 0.5 ]  = 1
    selected_g[ p_links <= 0.5 ] = 0	

    cat( paste( "" ), fill = TRUE )
    cat( paste( "Adjacency matrix of selected graph" ), fill = TRUE )
    cat( paste( "" ), fill = TRUE )
    
    Matrix::printSpMatrix( Matrix( selected_g, sparse = TRUE ), col.names = TRUE, note.dropping.colnames = FALSE )
    cat( paste( "" ), fill = TRUE )
    cat( paste( "Size of selected graph = ", sum( selected_g ) ), fill = TRUE )

    cat( paste( "Edge posterior probability of selected graph = ", 0.5 ), fill = TRUE )

    cat( paste( "" ), fill = TRUE )
} 
   
