% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GIRFandGFEVD.R, R/plotMethods.R,
%   R/printMethods.R
\name{GFEVD}
\alias{GFEVD}
\alias{plot.gfevd}
\alias{print.gfevd}
\title{Estimate generalized forecast error variance decomposition for structural
 STVAR models.}
\usage{
GFEVD(
  stvar,
  N = 30,
  shock_size = 1,
  initval_type = c("data", "random", "fixed"),
  which_cumulative = numeric(0),
  use_data_shocks = FALSE,
  data_gfevd_pars = c(0, 0.75),
  init_regime = 1,
  init_values = NULL,
  R1 = 250,
  R2 = 250,
  ncores = 2,
  burn_in = 1000,
  exo_weights = NULL,
  seeds = NULL,
  use_parallel = TRUE
)

\method{plot}{gfevd}(x, ..., data_shock_pars = NULL)

\method{print}{gfevd}(x, ..., digits = 2, N_to_print)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'}, created by, e.g., \code{fitSTVAR} or \code{fitSSTVAR}.}

\item{N}{a positive integer specifying the horizon how far ahead should the GFEVD be calculated.}

\item{shock_size}{What sign and size should be used for all shocks? By the normalization, the conditional
covariance matrix of the structural error is an identity matrix.}

\item{initval_type}{What type initial values are used for estimating the GIRFs that the GFEVD is based on?
\describe{
  \item{\code{"data"}:}{Estimate the GIRF for all the possible length \eqn{p} histories in the data.}
  \item{\code{"random"}:}{Estimate the GIRF for several random initial values generated from the a specific regime
     specified by the argument \code{init_regimes}. The number of initial values is set with the argument \code{R2}.}
  \item{\code{"fixed"}:}{Estimate the GIRF for the initial values specified with the argument \code{init_values}.}
}}

\item{which_cumulative}{a numeric vector with values in \eqn{1,...,d}
(\code{d=ncol(data)}) specifying which the variables for which the impulse
responses should be cumulative. Default is none.}

\item{use_data_shocks}{set \code{TRUE} (\strong{recommended}) for a special feature in which for every possible length \eqn{p} history
in the data, or a subset of them if so specified in the argument \code{data_gfevd_pars}, the GFEVD is estimated for a shock that
has the sign and size of the corresponding structural shock recovered from the data. See the details section.}

\item{data_gfevd_pars}{a length two numeric vector with the following elements determining settings for \code{initval_type="data"}
and \code{use_data_shocks=TRUE}:
\enumerate{
  \item An integer between \code{0} and \code{M} determining the (dominant) regime for which the GFEVD should be calculated (\code{0}
    for all regimes).
  \item A number between \code{0.5} and \code{1} determining how large transition weight a regime should have to be considered dominant
    in a given time period (i.e., determining which histories are used to calculate the GFEVD if the first element is not \code{0}).
}}

\item{init_regime}{an integer in \eqn{1,...,M} specifying the regime from which
the initial values should be generated from (using a simulation procedure with a burn-in period).
For models with Gaussian conditional distribution, it is also possible to generate the starting
values from the stationary distribution of a regime. See the details section.}

\item{init_values}{a size \code{[p, d, R2]} array specifying the initial values in each slice
for each Monte Carlo repetition, where d is the number of time series in the system and \code{R2}
is an argument of this function. In each slice, the \strong{last} row will be used as initial values
for the first lag, the second last row for second lag etc. If not specified, initial values will be
drawn from the regime specified in \code{init_regimes}.}

\item{R1}{the number of repetitions used to estimate GIRF for each initial
value.}

\item{R2}{the number of initial values to be drawn/used if \code{initval_type="random"} or \code{"fixed"}.}

\item{ncores}{the number CPU cores to be used in parallel computing. Only
single core computing is supported if an initial value is specified (and
the GIRF won't thus be estimated multiple times).}

\item{burn_in}{Burn-in period for simulating initial values from a regime.}

\item{exo_weights}{if \code{weight_function="exogenous"}, provide a size
\eqn{(N+1 \times M)} matrix of exogenous transition weights for the regimes: \code{[h, m]}
for the (after-the-impact) period \eqn{h-1} and regime \eqn{m} weight (\code{[1, m]}
is for the impact period). Ignored if \code{weight_function != "exogenous"}.}

\item{seeds}{a numeric vector containing the random number generator seed for estimation
of each GIRF. Should have the length of at least...
\itemize{
  \item ...\code{nrow(data) - p + 1} if \code{initval_type="data"} or \code{use_data_shocks=TRUE}.
  \item ...\code{R2} if \code{initval_type="random"}.
  \item ...\code{1} if \code{initval_type="fixed."}.
}
If the length of the vector is greater than what is needed, the extra seeds are dropped from the
end of the \code{seeds} vector. Set to \code{NULL} for not initializing the seed.}

\item{use_parallel}{employ parallel computing? If \code{FALSE}, does not print
anything.}

\item{x}{object of class \code{'gfevd'} generated by the function \code{GFEVD}.}

\item{...}{graphical parameters passed to the \code{'ts'} plot method when using \code{data_shock_pars}.}

\item{data_shock_pars}{if \code{use_data_shocks}, alternative plot method can be used that
plots the relative contribution of a given shock to the forecast error variance of each variable
at a given horizon. Should be a length two numeric vector with the number of the shock (1,..,d)
in the first element and the horizon (0,1,2,...,N) in the second element.}

\item{digits}{the number of decimals to print}

\item{N_to_print}{an integer specifying the horizon how far to print the estimates.
The default is that all the values are printed.}
}
\value{
Returns and object of class 'gfevd' containing the GFEVD for all the variables and to
  the transition weights. Note that the decomposition does not exist at horizon zero for transition weights
  because the related GIRFs are always zero at impact.
  Also contains the individual GFEVDs for each used initial length \eqn{p} initial value (history) as
  4D array with dimensions \code{[horizon, variable, shock, time]}.
}
\description{
\code{GFEVD} estimates generalized forecast error variance decomposition
 for structural STVAR models.
}
\details{
The GFEVD is a forecast error variance decomposition calculated with the generalized
  impulse response function (GIRF). See Lanne and Nyberg (2016) for details.

  If \code{use_data_shocks=TRUE}, each GIRF in the GFEVD is estimated for a shock that has the sign and size of the
  corresponding structural shock recovered from the fitted model. This is done for every possible length \eqn{p} history
  in the data, or to a subset of the histories based in the settings given in the argument \code{data_gfevd_pars}.
  The GFEVD is then calculated as the average of the GFEVDs obtained from the GIRFs estimated for
  the data shocks. The plot and print methods can be used as usual for this GFEVD, but there is also a special feature
  that allows to plot the contribution of each shock to the variance of the forecast errors at various horizons in specific
  historical points of time. This can be done by using the plot method with the argument \code{data_shock_pars}.
  Note that the arguments \code{shock_size}, \code{initval_type}, and \code{init_regime} are ignored if \code{use_data_shocks=TRUE}.
}
\section{Functions}{
\itemize{
\item \code{plot(gfevd)}: plot method

\item \code{print(gfevd)}: print method

}}
\examples{
 \donttest{
 # These are long-running examples that use parallel computing.
 # It takes approximately 30 seconds to run all the below examples.
 # Note that larger R1 and R2 should be used for more reliable results;
 # small R1 and R2 are used here to shorten the estimation time.

 # Recursively identifed logistic Student's t STVAR(p=3, M=2) model with the first
 # lag of the second variable as the switching variable:
 params32logt <- c(0.5959, 0.0447, 2.6279, 0.2897, 0.2837, 0.0504, -0.2188, 0.4008,
  0.3128, 0.0271, -0.1194, 0.1559, -0.0972, 0.0082, -0.1118, 0.2391, 0.164, -0.0363,
  -1.073, 0.6759, 3e-04, 0.0069, 0.4271, 0.0533, -0.0498, 0.0355, -0.4686, 0.0812,
   0.3368, 0.0035, 0.0325, 1.2289, -0.047, 0.1666, 1.2067, 7.2392, 11.6091)
 mod32logt <- STVAR(gdpdef, p=3, M=2, params=params32logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="recursive")

 # GFEVD for one-standard-error positive structural shocks, N=30 steps ahead,
 # with fix initial values assuming all possible histories in the data.
 gfevd1 <- GFEVD(mod32logt, shock_size=1, N=30, initval_type="data", R1=10,
   seeds=1:(nrow(mod32logt$data)-2))
 print(gfevd1) # Print the results
 plot(gfevd1) # Plot the GFEVD

 # GFEVD for one-standard-error positive structural shocks, N=30 steps ahead,
 # with fix initial values that are the last p observations of the data.
 gfevd2 <- GFEVD(mod32logt, shock_size=1, N=30, initval_type="fixed", R1=100, R2=1,
  init_values=array(mod32logt$data[(nrow(mod32logt$data) - 2):nrow(mod32logt$data),],
  dim=c(3, 2, 1)), seeds=1)
 plot(gfevd2) # Plot the GFEVD

 # GFEVD for two-standard-error negative structural shocks, N=50 steps ahead
 # with the inital values drawn from the first regime. The responses of both
 # variables are accumulated.
 gfevd3 <- GFEVD(mod32logt, shock_size=-2, N=50, initval_type="random",
  R1=50, R2=50, init_regime=1)
 plot(gfevd3) # Plot the GFEVD

 # GFEVD calculated for each lenght p history in the data in such a way that
 # for each history, the structural shock recovered from the fitted model is
 # used.
 gfevd4 <- GFEVD(mod32logt, N=20, use_data_shocks=TRUE, R1=10)
 plot(gfevd4) # The usual plot method

 # Plot the contribution of the first to the variance of the forecast errors at
 # the historial points of time using the structural shocks recovered from the data:
 plot(gfevd4, data_shock_pars=c(1, 0)) # Contribution at impact
 plot(gfevd4, data_shock_pars=c(1, 2)) # Contribution after two periods
 plot(gfevd4, data_shock_pars=c(1, 4)) # Contribution after four periods

 # GFEVD calculated for each length p history in the data in such a way that
 # for each history, the structural shock recovered from the fitted model is
 # used, and only include the histories in which Regime 1 is dominant (its
 # transition weight is at least 0.75):
 gfevd5 <- GFEVD(mod32logt, N=20, use_data_shocks=TRUE, data_gfevd_pars=c(1, 0.75),
   R1=10)
 plot(gfevd5) # Plot the GFEVD
 }
}
\references{
\itemize{
   \item Lanne M. and Nyberg H. 2016. Generalized Forecast Error Variance Decomposition for Linear
     and Nonlineae Multivariate Models. \emph{Oxford Bulletin of Economics and Statistics}, \strong{78}, 4, 595-603.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{linear_IRF}}, \code{\link{hist_decomp}}, \code{\link{cfact_hist}}, \code{\link{cfact_fore}},
 \code{\link{cfact_girf}}, \code{\link{fitSSTVAR}}
}
