% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stamp.shape.r
\name{stamp.shape}
\alias{stamp.shape}
\title{Compute shape indices on \code{stamp} output}
\usage{
stamp.shape(T1, T2, stmp, index = "PAR")
}
\arguments{
\item{T1}{a \code{SpatialPolygons} object of polygons from time 1.}

\item{T2}{a \code{SpatialPolygons} object of polygons from time 2.}

\item{stmp}{output \code{SpatialPolygonsDataFrame} generated from the \code{stamp} function.}

\item{index}{a character item identifying which shape metric is to be computed. See \bold{Details}.}
}
\value{
A \code{DataFrame} with four columns:
 \cr
 \code{GROUP} -- STAMP polygon groups from the \code{stamp} function.
 \code{T1.INDEX} -- shape index value for T1 polygons for each group. \code{INDEX} is replaced by name of index.
 \code{T2.INDEX} -- shape index value for T2 polygons for each group. \code{INDEX} is replaced by name of index.
 \code{d.INDEX} -- change (t2 - t1) in shape value for each group. \code{INDEX} is replaced by name of index.
}
\description{
This function computes a suite of shape complexity metrics on STAMP polygons
 facilitating shape analysis.
}
\details{
The \code{stamp.shape} function can be used to perform polygon shape analysis on output
 polygons from function \code{stamp}. Shape indices are computed on each output polygon.
 Five shape indices are available:
 \cr
 \code{"PER"} -- Shape perimeter, in appropriate units.
   \cr
 \code{"PAR"} -- Perimeter-area ratio, in appropriate units; \deqn{\mathtt{PAR} = \frac{p}{a}}{PAR=p/a}
   \cr
 \code{"FRAC"} -- Fractal dimension (Mandelbrot 1977, Lovejoy 1982); \deqn{\mathtt{FRAC} =\frac{2\log(p)}{\log(a)}}{FRAC=2log(p)/log(a)}
   \cr
 \code{"SHPI"} -- Shape index (Patton 1975); \deqn{\mathtt{SHPI} = \frac{p}{2*\sqrt{\pi*a}}}{SHPI=(p)/2*sqrt(pi*a)}
   \cr
 \code{"LIN"} -- Linearity index  (Baker and Cai 1992); \deqn{\mathtt{LIN} = 1 - \frac{a}{a_{circ}}}{LIN=1-(a/a_circ)}
   \cr
 Where \eqn{a} is polygon area, \eqn{p} is polygon perimeter, and \eqn{a_{circ}}{a_circ} is the
 area of the circumscribing (encompassing) circle of a polygon.
 \cr \cr
 Some Notes:
 \cr
 PER is simply the length of the perimeter, and is not an overly useful measure of shape, but may be useful
 in direct comparisons. PAR > 0, without limit with larger values sugesting more complex, irregular shapes.
 The range of FRAC is [1, 2]. FRAC approaches 1 for very simple shapes (squares, circles, etc.) and approaches
 2 for complex, convoluted shapes. SHPI > 1 without limit, as SHPI increase, the complexity of the shape 
 increases. The range of LIN is [0, 1]. A perfect circle will have a LIN of 0, while more linear shapes will approach 1.
 \cr
 \cr
 The indices PAR, FRAC, SHPI, and LIN are all essentially measures of shape complexity. LIN is unique in that it 
 tries to focus on the  linearity of the shape by comparing the area to a circle. LIN is however, less useful with
 STAMP events containing multiple polygons, as the calculation for the circumscribing circle will include all
 polygon objects within the group and artificially increase the LIN scores.
}
\examples{
library(sp)
library(rgeos)
library(raster)
data("fire1")
data("fire2")
#set globally unique ID column required for stamp function
fire1$ID <- 1:nrow(fire1) 
#set globally unique ID column required for stamp function
fire2$ID <- (max(fire1$ID)+1):(max(fire1$ID) + nrow(fire2)) 
ch <- stamp(fire1, fire2, dc=1, direction=FALSE, distance=FALSE)
ch.sh <- stamp.shape(T1 = fire1, T2 = fire2, stmp = ch, index = 'LIN')
}
\references{
Baker, W.L. and Cai, Y. (1992) The r.le programs for multiscale analysis of landscape structure using
   the GRASS geographical information system. \emph{Landscape Ecology}, 7(4):291-302. \cr\cr
 Lovejoy, S. (1982) Area-perimeter relation for rain and cloud areas. \emph{Science}, 216(4542):185-187.  \cr\cr
 Mandlebrot, B.B. (1977) Fractals, Form, Chance and Dimension. W.H Freeman and Co., New York.  \cr\cr
 Patton, D.R. (1977) A diversity index for quantifying habitat "edge". \emph{Wildlife Society Bulletin}, 3:171-173.
}
\seealso{
stamp
}
\keyword{stamp}
