% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/standardize.R
\name{standardize}
\alias{standardize}
\title{Standardize a formula and data frame for regression.}
\usage{
standardize(formula, data, family = gaussian, scale = 1,
  na.action = "na.pass")
}
\arguments{
\item{formula}{A regression \code{\link[stats]{formula}}.}

\item{data}{A data.frame containing the variables in \code{formula}.}

\item{family}{A regression \code{\link[stats]{family}} (default gaussian).}

\item{scale}{The desired scale for the regression frame. Must be a single
positive number. See 'Details'.}

\item{na.action}{See \code{\link[stats]{model.frame}}.}
}
\value{
A \code{\link[=standardized-class]{standardized}} object. The
  \code{formula} and \code{data} elements of the object can be used in calls
  to regression functions.
}
\description{
Create a \code{\link[=standardized-class]{standardized}} object which places
all variables in \code{data} on the same scale based on \code{formula},
making regression output easier to interpret.
For mixed effects regressions, this also offers computational benefits, and
for Bayesian regressions, it also makes determining reasonable priors easier.
}
\details{
First \code{\link[stats]{model.frame}} is called. Then,
if \code{family = gaussian}, the response is checked to ensure that it is
numeric and has more than two unique values.  If \code{\link{scale_by}} is
used on the response in \code{formula}, then the \code{scale} argument to
\code{scale_by} is ignored and forced to \code{1}.  If \code{\link{scale_by}}
is not called, then \code{\link[base]{scale}} is used with default arguments.
The result is that gaussian responses are on unit scale (i.e. have mean
\code{0} and standard deviation \code{1}), or, if \code{\link{scale_by}} is
used on the left hand side of \code{formula}, unit scale within each
level of the specified conditioning factor.
For all other values for \code{family}, the response is not checked.

For the predictors in the formula, first any random effects grouping factors
in the formula are coerced to factor and unused levels are dropped.  The
levels of the resulting factor are then recorded in the \code{predvars} terms
attribute.  Then for the remaining predictors, regardless of their original
class, if they have only two unique non-\code{NA} values, they are coerced
to unordered factors.  Then, \code{\link{named_contr_sum}} and
\code{\link{scaled_contr_poly}} are called for unordered and ordered factors,
respectively, using the \code{scale} argument provided in the call
to \code{standardize} as the \code{scale} argument to the contrast
functions.  For numeric variables, if the variable contains a call to
\code{\link{scale_by}}, then, regardless of whether the call to
\code{\link{scale_by}} specifies \code{scale}, the value of \code{scale}
in the call to \code{standardize} is used.  If the numeric variable
does not contain a call to \code{\link{scale_by}}, then
\code{\link[base]{scale}} is called, ensuring that the result has
standard deviation \code{scale}.

With the default value of \code{scale = 1}, the result is a
\code{\linkS4class{standardized}} object which contains a formula and data
frame which can be used to fit regressions where the predictors are all
on a similar scale.  Its data frame
has numeric variables on unit scale, unordered factors with named sum
sum contrasts, and ordered factors with orthogonal polynomial contrasts
on unit scale.  For gaussian regressions, the response is also placed on
unit scale.  If \code{scale = 0.5} (for example),
then gaussian responses would still
be placed on unit scale, but unordered factors' named sum contrasts would
take on values {-0.5, 0, 0.5} rather than {-1, 0, 1}, the standard deviation
of each column in the contrast matrices for ordered factors would be
\code{0.5} rather than \code{1}, and the standard deviation of numeric
variables would be \code{0.5} rather than \code{1} (within-factor-level
in the case of \code{\link{scale_by}} calls).
}
\section{Note}{
 Offsets are not currently supported. The \code{\link{scale_by}}
  function is supported so long as it is not nested within other function
  calls.  The \code{\link[stats]{poly}} function is supported so long as
  it is either not nested within other function calls, or is nested as the
  transformation of the numeric variable in a \code{\link{scale_by}} call.
  If \code{\link[stats]{poly}} is used, then the \code{lsmeans} function
  will yield misleading results (as would normally be the case).
}
\examples{
dat <- expand.grid(ufac = letters[1:3], ofac = 1:3)
dat <- as.data.frame(lapply(dat, function(n) rep(n, 60)))
dat$ofac <- factor(dat$ofac, ordered = TRUE)
dat$x <- rpois(nrow(dat), 5)
dat$z <- rnorm(nrow(dat), rep(rnorm(30), each = 18), rep(runif(30), each = 18))
dat$subj <- rep(1:30, each = 18)
dat$y <- rnorm(nrow(dat), -2, 5)

sdat <- standardize(y ~ log(x + 1) + scale_by(z ~ subj) + ufac + ofac +
  (1 | subj), dat)

sdat
sdat$formula
head(dat)
head(sdat$data)
sdat$contrasts
sdat$groups
mean(sdat$data$y)
sd(sdat$data$y)
mean(sdat$data$log_x.p.1)
sd(sdat$data$log_x.p.1)
with(sdat$data, tapply(scale_z_by_subj, subj, mean))
with(sdat$data, tapply(scale_z_by_subj, subj, sd))

sdat <- standardize(y ~ log(x + 1) + scale_by(z ~ subj) + ufac + ofac +
  (1 | subj), dat, scale = 0.5)

sdat
sdat$formula
head(dat)
head(sdat$data)
sdat$contrasts
sdat$groups
mean(sdat$data$y)
sd(sdat$data$y)
mean(sdat$data$log_x.p.1)
sd(sdat$data$log_x.p.1)
with(sdat$data, tapply(scale_z_by_subj, subj, mean))
with(sdat$data, tapply(scale_z_by_subj, subj, sd))

\dontrun{
mod <- lmer(sdat$formula, sdat$data)
# this next line causes warnings about contrasts being dropped, but
# these warnings can be ignored (i.e. the statement still evaluates to TRUE)
all.equal(predict(mod, newdata = predict(sdat, dat)), fitted(mod))
}

}
\seealso{
\code{\link[base]{scale}}, \code{\link{scale_by}},
  \code{\link{named_contr_sum}}, and \code{\link{scaled_contr_poly}}.
}

