% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statGraph.R
\name{GIC}
\alias{GIC}
\title{Graph Information Criterion (GIC)}
\usage{
GIC(
  G,
  model,
  p = NULL,
  bandwidth = "Silverman",
  eigenvalues = NULL,
  dist = "KL"
)
}
\arguments{
\item{G}{the undirected graph (igraph type) or its adjacency matrix. The
adjacency matrix of an unweighted graph contains only 0s and 1s, while the
weighted graph may have nonnegative real values that correspond to the
weights of the edges.}

\item{model}{either a list, a string, a function or a matrix describing a
graph model:

A list that represents the spectral density of a model. It contains the
components "x" and "y", which are numeric vectors of the same size. The "x"
component contains the points at which the density was computed and the "y"
component contains the observed density.

A string that indicates one of the following models: "ER" (Erdos-Renyi random
graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS"
(Watts-Strogatz model), and "BA" (Barabasi-Albert model). When the argument
'model' is a string, the user must also provides the model parameter by using
the argument 'p'.

A function that returns a graph (represented by its adjacency matrix)
generated by a graph model. It must contain two arguments: the first one
corresponds to the graph size and the second to the parameter of the model.
The model parameter will be provided by the argument 'p' of the 'GIC'
function.

A matrix containing the spectrum of the model. Each column contains the
eigenvalues of a graph generated by the model. To estimate the spectral
density of the model, the method will estimate the density of the values of
each column, and then will take the average density.}

\item{p}{the model parameter. The user must provide a valid parameter if the
argument 'model' is a string or a function.
For the predefined models ("ER", "GRG", "KR", "WS", and "BA"), the parameter
the probability to connect a pair of vertices, for the "ER" model
(Erdos-Renyi random graph);

the radius used to construct the geometric graph in a unit square, for the
"GRG" model (geometric random graph);

the degree 'k' of a regular graph, for the "KR" model (k regular random
graph);

the probability to reconnect a vertex, for the "WS" model (Watts-Strogatz
model);

and the scaling exponent, for the "BA" model (Barabasi-Albert model).}

\item{bandwidth}{string showing which criterion is used to choose the
bandwidth during the spectral density estimation. Choose between the
following criteria: "Silverman" (default), "Sturges", "bcv", "ucv" and "SJ".
"bcv" is an abbreviation of biased cross-validation, while "ucv" means
unbiased cross-validation. "SJ"  implements the methods of Sheather & Jones
(1991) to select the bandwidth using pilot estimation of derivatives.}

\item{eigenvalues}{optional parameter. It contains the eigenvalues of matrix
G. Then, it can be used when the eigenvalues of G were previously computed.
If no value is passed, then the eigenvalues of G will be computed by 'GIC'.}

\item{dist}{string indicating if you want to use the "KL" (default) or "L2"
distances. "KL" means Kullback-Leibler divergence.}
}
\value{
A real number corresponding to the Kullback-Leibler divergence or L2
distance between the observed graph and the graph model.
}
\description{
\code{GIC} returns the Kullback-Leibler divergence or L2 distance between an
undirected graph and a given graph model.
}
\examples{
set.seed(1)
G <- as.matrix(igraph::get.adjacency(igraph::sample_gnp(n=50, p=0.5)))

# Using a string to indicate the graph model
result1 <- GIC(G, "ER", 0.5)
result1

# Using a function to describe the graph model
# Erdos-Renyi graph
model <- function(n, p) {
   return(as.matrix(igraph::get.adjacency(igraph::sample_gnp(n, p))))
}
result2 <- GIC(G, model, 0.5)
result2
}
\references{
Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
Discriminating Different Classes of Biological Networks by Analyzing the
Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
doi:10.1371/journal.pone.0049949.

Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.

Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
*21*, 65-66.

Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
selection method for kernel density estimation.
_Journal of the Royal Statistical Society series B_, 53, 683-690.
http://www.jstor.org/stable/2345597.
}
\keyword{graph_information_criterion}
