% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RFcv2.R
\name{RFcv2}
\alias{RFcv2}
\title{Cross validation, n-fold for random forest (RF)}
\usage{
RFcv2(
  trainx,
  trainy,
  cv.fold = 10,
  mtry = if (!is.null(trainy) && !is.factor(trainy)) max(floor(ncol(trainx)/3), 1) else
    floor(sqrt(ncol(trainx))),
  ntree = 500,
  predacc = "VEcv",
  ...
)
}
\arguments{
\item{trainx}{a dataframe or matrix contains columns of predictor variables.}

\item{trainy}{a vector of response, must have length equal to the number of
rows in trainx.}

\item{cv.fold}{integer; number of folds in the cross-validation. if > 1,
then apply n-fold cross validation; the default is 10, i.e., 10-fold cross
validation that is recommended.}

\item{mtry}{a function of number of remaining predictor variables to use as
the mtry parameter in the randomForest call.}

\item{ntree}{number of trees to grow. This should not be set to too small a
number, to ensure that every input row gets predicted at least a few times.
By default, 500 is used.}

\item{predacc}{"VEcv" for vecv for numerical data, or "ccr" (i.e., correct
classification rate) or "kappa" for categorical data.}

\item{...}{other arguments passed on to randomForest.}
}
\value{
A list with the following component:
vecv for numerical data: ; or
 ccr (correct classification rate) for categorical data: .
}
\description{
This function is a cross validation function
for random forest. It is for functions 'steprf', 'steprfAVI', ect.
}
\note{
This function is largely based on rf.cv (see Li et al. 2013) and
rfcv in randomForest.
}
\examples{
\donttest{
library(spm)
data(hard)
data(petrel)

rfcv1 <- RFcv2(petrel[, c(1,2, 6:9)], petrel[, 5], predacc = "VEcv")
rfcv1

rfcv2 <- RFcv2(hard[, -c(1, 17)], hard[, 17], predacc = "ccr")
rfcv2

rfcv3 <- RFcv2(hard[, -c(1, 17)], hard[, 17], predacc = "kappa")
rfcv3

n <- 10 # number of iterations, 60 to 100 is recommended.
VEcv <- NULL
for (i in 1:n) {
rfcv1 <- RFcv2(petrel[, c(1,2,6:9)], petrel[, 5], predacc = "VEcv")
VEcv [i] <- rfcv1
}
plot(VEcv ~ c(1:n), xlab = "Iteration for RF", ylab = "VEcv (\%)")
points(cumsum(VEcv) / c(1:n) ~ c(1:n), col = 2)
abline(h = mean(VEcv), col = 'blue', lwd = 2)

n <- 10 # number of iterations, 60 to 100 is recommended.
measures <- NULL
for (i in 1:n) {
rfcv1 <- RFcv2(hard[, c(4:6)], hard[, 17], predacc = "ccr")
measures <- rbind(measures, rfcv1)
}
plot(measures ~ c(1:n), xlab = "Iteration for RF", ylab = "Correct
classification Rate  (\%)")
points(cumsum(measures) / c(1:n) ~ c(1:n), col = 2)
abline(h = mean(measures), col = 'blue', lwd = 2)
}

}
\references{
Li, J., J. Siwabessy, M. Tran, Z. Huang, and A. Heap. 2013.
Predicting Seabed Hardness Using Random Forest in R. Pages 299-329 in Y.
Zhao and Y. Cen, editors. Data Mining Applications with R. Elsevier.

Li, J. 2013. Predicting the spatial distribution of seabed gravel content
using random forest, spatial interpolation methods and their hybrid methods.
Pages 394-400  The International Congress on Modelling and Simulation
(MODSIM) 2013, Adelaide.

Liaw, A. and M. Wiener (2002). Classification and Regression by
randomForest. R News 2(3), 18-22.
}
\author{
Jin Li
}
